/*

  SmartClient Ajax RIA system
  Version SNAPSHOT_v15.0d_2025-11-04/LGPL Deployment (2025-11-04)

  Copyright 2000 and beyond Isomorphic Software, Inc. All rights reserved.
  "SmartClient" is a trademark of Isomorphic Software, Inc.

  LICENSE NOTICE
     INSTALLATION OR USE OF THIS SOFTWARE INDICATES YOUR ACCEPTANCE OF
     ISOMORPHIC SOFTWARE LICENSE TERMS. If you have received this file
     without an accompanying Isomorphic Software license file, please
     contact licensing@isomorphic.com for details. Unauthorized copying and
     use of this software is a violation of international copyright law.

  DEVELOPMENT ONLY - DO NOT DEPLOY
     This software is provided for evaluation, training, and development
     purposes only. It may include supplementary components that are not
     licensed for deployment. The separate DEPLOY package for this release
     contains SmartClient components that are licensed for deployment.

  PROPRIETARY & PROTECTED MATERIAL
     This software contains proprietary materials that are protected by
     contract and intellectual property law. You are expressly prohibited
     from attempting to reverse engineer this software or modify this
     software for human readability.

  CONTACT ISOMORPHIC
     For more information regarding license rights and restrictions, or to
     report possible license violations, please contact Isomorphic Software
     by email (licensing@isomorphic.com) or web (www.isomorphic.com).

*/
// ----------------------------------------------------------------------------------------

// If ListGrid, or DynamicForm isn't loaded don't attempt to create this class - it's a requirement.
if (isc.ListGrid != null && isc.DynamicForm != null) {

// Utility class for picking foreignKey and includeFrom properties in the DataSourceEditor.
// Provide validDsNames on creation (or set them when showing the picker).
// Fires "changed(form, value)" when the combinedValue changes, returning it as dsName.fieldName
isc.ClassFactory.defineClass("DataSourceFieldPicker", "DynamicForm");

isc.DataSourceFieldPicker.addClassProperties({
    // The warning text where we can't guess foreign keys. 
    FOREIGN_KEY_WARNING: "Could not guess which foreignKey to use. " +
                          "Determine which of your fields is the foreign key, " +
                          "and make its foreignKey property point to a field in ",
                          
    getForeignKeyWarning : function(foreignDsName) {
        return isc.DataSourceFieldPicker.FOREIGN_KEY_WARNING + "'" + foreignDsName + "'.";
    }
});

isc.DataSourceFieldPicker.addProperties({
    // An array of valid DataSource names. The DataSources need not be loaded -- the picker
    // will lazily call isc.DS.load when necessary to get field names.
    // validDsNames: null,

    // An array of records representing all known DataSources, even if we don't know that they 
    // are valid yet. If chosen, they can be lazily validated via a callback.
    // If both validDsNames and allDSNames are provided, then we will offer both lists in the
    // drop down menu, with a separator in between.
    // allDsRecords: null,

    // The required base type of the field. Leave null if any field type is fine.
    // requiredBaseType: null,

    // Try to guess a foreign key with this datasource record. The record should contain
    // at least an ID and an array of field records
    // warnIfNoForeignKey: null,

    // The value of the dsName and fieldName, in the form dsName.fieldName
    // combinedValue: "",

    fields: [{
        name: "DataSource",
        type: "Select",

        // Restore default titleStyle, because the tool skin doesn't seem to
        // be applied consistently in the picker
        titleStyle: "formTitle",
        
        // Allow empty values so that the user can choose no DataSourceField
        allowEmptyValue: true,
       
        valueField: "ID",
        getClientPickListData : function() {
            return this.form.getDatasourcePickListData();
        },

        changed : function(form, item, value) {
            form.handleDsNameChanged(value);
        }
    },{
        name: "Field",
        type: "Select",
        
        // Restore default titleStyle, because the tool skin doesn't seem to
        // be applied consistently in the picker
        titleStyle: "formTitle",
        
        changed : function(form, item, value) {
            form.handleChanged();
        }
    }],

    getDatasourcePickListData : function () {
        // Setting validDsNames or allDsRecords will reset _datasourcePickListData, causing a
        // lazy recalculation here.
        if (!this._datasourcePickListData) {
            // Always allow an empty value!
            this._datasourcePickListData = [""];
            
            if (this.validDsNames && this.validDsNames.getLength() > 0) {
                this._datasourcePickListData.addList(this.validDsNames.map(function (dsName) {
                    return {ID: dsName};
                }));
            }

            if (this.allDsRecords && this.allDsRecords.getLength() > 0) {
                if (this._datasourcePickListData.getLength() > 0) {
                    this._datasourcePickListData.add({isSeparator: true});
                }

                // allDsRecords is already an array of records ...
                this._datasourcePickListData.addList(this.allDsRecords);
            }
        }

        return this._datasourcePickListData;
    },

    setValidDsNames : function(validDsNames) {
        this.validDsNames = validDsNames;
        this._datasourcePickListData = null;
    },

    setAllDsRecords : function(allDsRecords) {
        this.allDsRecords = allDsRecords;
        this._datasourcePickListData = null;
    },

    setWarnIfNoForeignKey : function(dsRecord) {
        this.warnIfNoForeignKey = dsRecord;
    },
    
    setCombinedValue : function(value) {
        var dsItem = this.getItem("DataSource");
        var fieldItem = this.getItem("Field");
        
        var parts = (value || "").split(".");
        dsItem.setValue(parts[0]);
        fieldItem.setValue(parts[1]);

        this.handleDsNameChanged(parts[0]);
    },

    getCombinedValue : function() {
        var value = this.getValue("DataSource");
        var fieldName = this.getValue("Field");
        if (fieldName) value = value + "." + fieldName;
        return value;
    },

    initWidget : function() {
        this.Super("initWidget", arguments);
        
        if (this.combinedValue) this.setCombinedValue(this.combinedValue);
    },

    _warnIfCannotGuessForeignKey : function (foreignDS) {
        var ourDsRec = this.warnIfNoForeignKey;
        
        if (!ourDsRec || !ourDsRec.fields) return;

        // If we have a field with a foreignKey defined that points to the foreignDS,
        // then we're fine.
        var foreignKeys = ourDsRec.fields.map(function (field) {
            return field.foreignKey ? field.foreignKey.split('.')[0] : null;
        });
        if (foreignKeys.contains(foreignDS.ID)) return;

        // We would also be fine if the foreignDS has a field with a foreignKey that
        // points back to us.
        foreignKeys = foreignDS.getFieldNames().map(function (fieldName) {
            var field = foreignDS.getField(fieldName);
            return field.foreignKey ? field.foreignKey.split('.')[0] : null;
        });
        if (foreignKeys.contains(ourDsRec.ID)) return;
    
        // We are also fine if there is a field in ourDsRec and foreignDsRec with
        // matching names
        var ourFieldNames = ourDsRec.fields.getProperty("name");
        var foreignFieldNames = foreignDS.getFieldNames();
        if (ourFieldNames.intersect(foreignFieldNames).getLength() > 0) return;
        
        // If we've gotten this far, then we can't guess the foreignKey. So, we'll
        // display a warning. Note that while we're using the standard error mechanism
        // for DyanmicForms, we don't actually prevent the update from occurring -- the
        // user can save this value if they like -- it's just a warning.
        this.addFieldErrors("DataSource", 
            isc.DataSourceFieldPicker.getForeignKeyWarning(this.getValue("DataSource")), 
            true);
    },

    // Callback when we load the live DS that corresponds to the dsName chosen
    // Can be called with null if loading the DS from the server has failed
    handleLiveDs : function(ds) {
        var fieldNames = [];
        var field = this.getField("Field");
        
        if (ds) {
            // Figure out the available field names
            fieldNames = ds.getFieldNames();
            if (this.requiredBaseType) {
                var self = this;
                fieldNames = fieldNames.findAll(function(field) {
                    var baseType = isc.SimpleType.getBaseType(ds.getField(field).type, ds);
                    return baseType == self.requiredBaseType;
                });
            }
            
            // If there is only one possible value, we may as well autodetect it
            if (fieldNames.getLength() == 1) {
                field.setValue(fieldNames[0]);
                this.handleChanged();
            }

            // If our current value isn't possible, then reset it
            if (!fieldNames.contains(field.getValue())) {
                field.setValue("");
                this.handleChanged();
            }

            // If we're checking for possible foreignKeys, then validate them
            if (this.warnIfNoForeignKey) this._warnIfCannotGuessForeignKey(ds);
        }

        field.setValueMap(fieldNames);
    },

    handleDsNameChanged : function(dsName) {
        // If value is empty, then may as well reset the field too
        if (!dsName) this.getField("Field").setValue("");
        
        // And fire the change event
        this.handleChanged();
       
        // Reset the value map for the fields. This will get filled in to
        // the available values when we get the live DS.
        this.getField("Field").setValueMap([]);
        
        // And reset any warnings about foreign keys -- that will also get
        // checked when we get the live DS.
        this.clearFieldErrors("DataSource", true);

        // Try to get the live DS that corresponds to the dsName chosen,
        // either synchronously or asynchronously
        if (dsName) {
            // Handle dataSources defined in the dsEditor session when useLiveDataSources=false
            if (this.session && this.session.useLiveDataSources === false) {
                const ds = this.session.get(dsName);
                if (ds) {
                } else {
                    this.logWarn("Loading dataSource from session was unsuccessful for " + dsName);
                }
                return;
            }
            // Otherwise, load the live DS asynchronously
            var ds = isc.DS.get(dsName);
            if (ds) {
                this.handleLiveDs(ds);
            } else {
                var self = this;
                isc.DS.load(dsName, function() {
                    ds = isc.DS.get(dsName);
                    if (!ds) self.logWarn("Loading dataSource from server was unsuccessful for " + dsName);
                    self.handleLiveDs(ds);
                }, false, true); 
            }
        }
    },

    handleChanged : function() {
        if (this.changed) this.changed(this, this.getCombinedValue());
    }
});

isc.DataSourceFieldPicker.registerStringMethods({
    // Fired when the combinedValue in the picker changes
    changed : "form, value"
});

//> @class DataSourceEditor
// Provides a UI for creating and editing +link{DataSource, DataSources}.
// <P>
// The DataSourceEditor uses a +link{dataSourceEditor.session,DataSourceEditorSession} to manage changes
// made during an editing session. Changes are tracked by the session and can be saved via the
// +link{DataSourceEditor.save,save()} method or by using the built-in save button
// (controlled by +link{dataSourceEditor.showActionButtons}).
// The editor will automatically create a session
// if you don't provide one explicitly via the +link{dataSourceEditor.session} property.
// <P>
// To use the DataSourceEditor, you must provide a +link{dataSourceEditor.dsDataSource} to handle loading and saving 
// DataSource XML definitions. This will typically be a +link{group:fileSource,fileSource dataSource},
// though this is not required (see +link{dataSourceEditorSession.dsDataSourceIsFileSource}).
// <P>
// If you set +link{DataSourceEditor.knownDataSources} to an array of
// +link{PaletteNode,PaletteNodes} representing all DataSources available in your project, the editor
// will allow the user to set up 
// +link{DataSourceField.includeFrom,included fields} and 
// +link{DataSourceField.foreignKey,relations between dataSources}.
// <P>
// For custom save workflows, handle the +link{dataSourceEditor.editComplete} notification and call 
// +link{dataSourceEditor.save()} to persist
// changes. Set +link{dataSourceEditor.showActionButtons} to false to hide the default UI buttons and 
// manage the workflow yourself.
//  
// @inheritsFrom VLayout
// @treeLocation Client Reference/Tools
// @visibility reifyOnSite
//<
isc.ClassFactory.defineClass("DataSourceEditor", "VLayout");

isc.DataSourceEditor.addClassMethods({
    // Returns the clean initialization data for a DataSource (the live data
    // contains various derived state) by loading the definition from the <dsDataSource>
    fetchDataSourceDefaults : function (dsDataSource, dsID, callback) {
        if (!dsDataSource) {
            this.logWarn("Cannot determine DS defaults because dsDataSource is not provided");
            return null;
        }

        var convertLiveInstance = function () {
            // serialize instance to XML
            var ds = isc.DS.get(dsID);
            if (!ds) {
                callback(dsID, null);
                return;
            }
            var dsClass = ds.getClassName(),
                schema
            ;

            if (isc.DS.isRegistered(dsClass)) {
                schema = isc.DS.get(dsClass);
            } else {
                schema = isc.DS.get("DataSource");
                ds._constructor = dsClass;
            }
            var xml = schema.xmlSerialize(ds);

            // and get JS the same as would be returned by getFile()
            isc.DMI.callBuiltin({
                methodName: "xmlToJS",
                "arguments": xml,
                callback : function (rpcResponse, data, rpcRequest) {
                    var data = rpcResponse.data;
                    if (!data) {
                        callback(dsID, null);
                        return;
                    }
                    var defaults = isc.DataSourceEditor.extractDSDefaultsFromJS(data, dsDataSource);

                    // ID, defaults, readOnly
                    callback(dsID, defaults, true);
                }
            });    
        };

        dsDataSource.getFile({
            fileName: dsID,
            fileType: "ds",
            fileFormat: "xml"
        }, function (dsResponse, data, dsRequest) {
            // data argument is the XML response. We want the JS response
            if (dsResponse.data.length == 0 || !dsResponse.data[0].fileContentsJS) {
                convertLiveInstance();
                return;
            }
            data = dsResponse.data[0].fileContentsJS;

            var defaults = isc.DataSourceEditor.extractDSDefaultsFromJS(data, dsDataSource);

            callback(dsID, defaults);
        }, {
            // DataSources are always shared across users
            // and we want JavaScript in the response to avoid an extra xmlToJs call
            operationId: "allOwnersXmlToJs"
        });
    },

    extractDSDefaultsFromJS : function (jsData, dsDataSource) {
        // Setup class instance creation so that any component creation attempt is captured
        // as "defaults" rather than creating an instance. 

        // Don't report DataSource collisions, enable capture defaults mode,
        // and record any global creations into a temporary windowContext instead of window[]
        isc.ClassFactory._setVBLoadingDataSources(true);
        isc.captureDefaults = true;
        isc.windowContext = {};

        //!OBFUSCATEOK
        var dsComponent = isc.eval(jsData);

        // Pull DS class type and push into defaults._constructor so the correct
        // schema will be serialized when saved.
        var defaults = dsComponent.defaults,
            dsID = defaults.ID || defaults.autoID,
            paletteNode = dsID && isc.windowContext[dsID],
            type = paletteNode && paletteNode.type
        ;
        if (!defaults._constructor) defaults._constructor = type;

        isc.windowContext = null;
        isc.captureDefaults = null;
        isc.ClassFactory._setVBLoadingDataSources(null);

        // Clear list of captured components so it is ready for the next capture
        isc.capturedComponents = null;

        // do some automatic defaulting otherwise done at DataSource.init()
        if (defaults.serverType == "sql") defaults.dataFormat = "iscServer";
        if (defaults.recordXPath != null && defaults.dataFormat == null) {
            defaults.dataFormat = "xml";
        }

        return defaults;
    },

    // Used by both DataSourceEditor and RelationEditor
    createNameFromTitle : function (title) {
        if (!title || title == "") return null;

        // Split title by disallowed characters
        var words = title.split(/[^a-zA-Z0-9_]/),
            parts = []
        ;
        for (var i = 0; i < words.length; i++) {
            var word = words[i];
            if (parts.length == 0) {
                // Initial word must start with a letter or underscore
                var m = word.match("^[0-9]*");
                if (m.length > 0) {
                    word = word.substring(m[0].length);
                }
                // and start with a lowercase letter
                word = word.substring(0,1).toLowerCase() + word.substring(1);
            } else {
                // Subsequence words start with uppercase
                word = word.substring(0,1).toUpperCase() + word.substring(1);
            }
            if (word && word != "") parts.add(word);
        }
        return parts.join('');
    }
});

isc.DataSourceEditor.addProperties({
// attributes 
overflow: "visible",

//> @attr dataSourceEditor.dsDataSource (DataSource | ID : null : IRW)
// The +link{dataSource} to be used to load and save dataSource XML. 
// <P>
// If no explicit +link{session} was provided, this property will be applied to the
// generated session object. If an explicit session was provided, the 
// the dsDataSource should be specified directly on the session.
//
// @visibility reifyOnSite
//<

//> @attr dataSourceEditor.knownDataSources (Array of PaletteNode : null : IR)
// A list of all known DataSources, as +link{paletteNode,PaletteNodes}, to be available 
// to the editor. These allow the dataSourceEditor to edit foreign keys, relations
// and includeFrom fields.
// <P>
// Each element of the array should include at least <code>ID</code> and <code>type</code>
// properties. If <code>defaults</code> are not included, these will be loaded from
// <P>
// If no explicit +link{session} was provided, this property will be applied to the
// generated session object. If an explicit session was provided, the 
// the knownDataSources should be specified directly on the session.
// +link{dsDataSource}.
//
// @visibility reifyOnSite
//<

//> @attr dataSourceEditor.useLiveDataSources (Boolean : null : IR)
// Determines whether +link{dataSourceEditorSession.get()} should return a
// live DataSource as obtained from +link{dataSource.get()}.
// <P>
// If set to false, a temporary DataSource
// will be created as needed from defaults that will not conflict with any
// live instance.
// <P>
// If no explicit +link{session} was provided, this property will be applied to the
// generated session object. If an explicit session was provided, the 
// useLiveDataSources should be specified directly on the session.
//
// @visibility reifyOnSite
//<

//> @attr dataSourceEditor.session (AutoChild DataSourceEditorSession : null : IR)
// The editor "session" that maintains the pending DataSource changes. A new session
// is automatically created during init if not provided.
// <P>
// When creating the initial session, the following DataSourceEditor properties
// are passed to the new session:
// <ul>
//   <li>+link{dsDataSource}</li>
//   <li>+link{knownDataSources} as +link{dataSourceEditorSession.dataSources}</li>
//   <li>+link{useLiveDataSources}</li>
// </ul>
//
// @visibility reifyOnSite
//<

sessionConstructor: "DataSourceEditorSession",

//> @attr dataSourceEditor.useChangeTracking (Boolean : null : IR)
// Should DataSource changes be tracked?
//
// @visibility internal
//<

//> @attr dataSourceEditor.readOnly (Boolean : null : IRW)
// Is this editor in read-only mode? In read-only mode, warnings normally shown
// for save issues (+link{validateForSave}) are suppressed because the result is not
// to be saved.
// <P>
// Normal interactions in the editor continue as usual.
// <P>
// This property is also applied to a +link{RelationEditor} that is shown.
//
// @visibility reifyOnSite
//<

//> @attr dataSourceEditor.enableRelationEditor (Boolean : null : IR)
// Should +link{relationEditor} be available while editing the DataSource?
// <P>
// Set to true to allow the user to select a field and show the relation editor
// to edit that field.
//
// @visibility reifyOnSite
//<

//> @attr dataSourceEditor.allowExplicitPKInSampleData (Boolean : null : IR)
// Set this property to true to allow a primary key to be specified for a field
// in an sample data DataSource definition.
//
// @visibility reifyOnSite
//<


//> @attr dataSourceEditor.instructions (HTMLString : null : IRW)
// Instructions to be shown within the editor above the DataSource ID.
//
// @setter setInstructions
// @group i18nMessages
// @visibility reifyOnSite
//<

//> @method dataSourceEditor.setInstructions
// Sets +link{instructions} to a new value and shows the instructions section.
// If set to <code>null</code> the instruction section will be hidden.
//
// @param [contents] (HTMLString) the instructions to show or null to hide section
// @visibility reifyOnSite
//<
setInstructions : function (contents) {
    if (!this.mainStack) return;
    if (contents) {
        this.mainStack.showSection(0);
        this.instructions.setContents(contents);
    } else { 
        this.mainStack.hideSection(0);
    }
},

// ---------------------------------------------------------------------------------------
// Auto-child defaults

//> @attr dataSourceEditor.mainEditor (AutoChild ComponentEditor : null : IRW)
// 
// Editor for dataSource properties
// @visibility reifyOnSite
//<
mainEditorDefaults: {
    _constructor: "ComponentEditor",
    autoDraw:false,
    autoFocus:true,
    numCols:8,
    colWidths:null,
    overflow:"visible",
    dataSource:"DataSource",
    itemHoverStyle: "docHover",
    titleHoverHTML : function (item) {
        if (isc.jsdoc.hasData()) {
            // the dataSource is the class
            var html = isc.jsdoc.hoverHTML("DataSource", item.name);
            if (html) return html;
        }
        // no doc exists for this attribute - show a hover with just the attribute name in bold so
        // the user doesn't wait forever for the tooltip
        return "<nobr><code><b>"+item.name+"</b></code> (no doc available)</nobr>";
    },
    initWidget : function () {
        if (this.useChangeTracking) {
            // Hook all fields except ID (which has its own handlers) to add handlers
            // to capture and record property changes for change tracking.
            var fields = this.fields;
            for (var i = 0; i < fields.length; i++) {
                var field = fields[i];
                if (field.name != "ID") {
                    isc.addProperties(field, {
                        editorEnter : function (form, item, value) {
                            this._origValue = value;
                        },
                        setValue : function (newValue) {
                            this._origValue = newValue;
                            return this.Super("setValue", arguments);
                        },
                        editorExit : function (form, item, value) {
                            var origValue = this._origValue;
                            if (origValue != value) {
                                form.creator.dataSourcePropertyChanged(item.name, origValue, value);
                                this._origValue = value;
                            }
                        }
                    });
                }
            }
        }
        this.Super("initWidget", arguments);
    }
},


getMainEditorFields : function () {
    var defaultFields = isc.clone(this.mainEditorFields);

    // Advanced usage: Allow arbitrary DS attributes to be added to the field editor
    if (this.extraEditorFields && this.extraEditorFields.length > 0) {
        var extraItemNames = this.extraEditorFields.getProperty("name");
        
        defaultFields.add({ type:"section", defaultValue:this.extraEditorFieldsTitle, itemIds:extraItemNames});
        defaultFields.addList(isc.clone(this.extraEditorFields));
    }
    return defaultFields
},

//> @attr dataSourceEditor.extraEditorFields (Array of FormItem : null : IRA)
// Optional extra fields to show in the field editor.
// <P>
// If specified as a non-empty array, these items will be displayed in the
// field editing form when the user selects the +link{moreButton,"More" button}.
// <P>
// The items will be added to a section at the end of the form with the
// +link{extraEditorFieldsTitle}.
// @visibility reifyOnSite
//<

//> @attr dataSourceEditor.extraEditorFieldsTitle (HTMLString : "Additional Properties" : IRA)
// Title for the section containing optional +link{extraEditorFields,extra items} in the 
// field editor.
// @visibility reifyOnSite
//<
extraEditorFieldsTitle:"Additional Properties",

//> @attr dataSourceEditor.showMainEditorSectionHeaders (boolean : true : IR)
// This option will suppress the +link{class:SectionItem,section headers} in the 
// +link{mainEditor} form for a lighter-weight appearance.
// @visibility reifyOnSite
//<
showMainEditorSectionHeaders:true,

mainEditorFields: [
    {name:"ID", title: "ID", 
        // ineffective in the presence of a validators array
        //required:true, 
        // this doesn't seem to work fully - possibly due to there being custom logic in 
        // editorEnter() and editorExit() - doesn't show the icon, for example
        validateOnExit: true, 
        // this works to enforce "required", including showing the icon properly
        validateOnChange: true, 
        hoverWidth: 300,
        selectOnFocus: true,
        editorEnter : function (form, item, value) {
            this._origValue = value;
        },
        setValue : function (newValue) {
            this._origValue = newValue;
            return this.Super("setValue", arguments);
        },
        editorExit : function (form, item, value) {
            var origValue = this._origValue;
            if (value && origValue != value) {
                form.creator.dataSourceIDChanged(origValue, value);
                this._origValue = value;
            }
        },
        validators: [
            { type: "required", stopIfFalse: true },
            { 
                type: "regexp",
                expression: "^(?!isc_).*$",
                errorMessage: "DataSource ID must not start with 'isc_'. That prefix is reserved for framework DataSources.'",
                stopIfFalse: true
            },
            { 
                type: "regexp",
                expression: "^[a-zA-Z_][a-zA-Z0-9_]*$",
                errorMessage: "DataSource ID must not contain spaces or punctuation other than underscore (_), and may not start with a number",
                stopIfFalse: true
            },
            {
                type:"custom",
                condition: function (item, validator, value, record, additionalContext) {
                    if (!value) return true;
                    if (!validator.idMap) {
                        // Create idMap to map from lowercase ID to actual ID so that
                        // entered name can be matched to an existing schema regardless
                        // of case.
                        var allDataSources = isc.DS.getRegisteredDataSourceObjects(),
                            idMap = {}
                        ;
                        for (var i = 0; i < allDataSources.length; i++) {
                            var ds = allDataSources[i];
                            if (ds && ds.componentSchema) {
                                var id = ds.ID;
                                idMap[id.toLowerCase()] = id;
                            }
                        }
                        validator.idMap = idMap;
                    }
                    // Also validate that the non-schema DS is not a SimpleType and not
                    // an internal application DS (i.e. a Reify-created DS is OK)
                    var id = validator.idMap[value.toLowerCase()] || value,
                        ds = item.form.creator.session.get(id),
                        isProjectDataSource = ds && (ds.sourceDataSourceID != null ||
                            item.form.creator.session.getDataSourceDefaults(id) != null)
                    ;
                    return ((!ds || (!ds.componentSchema && isProjectDataSource)) &&
                            !isc.SimpleType.getType(value));
                },
                errorMessage: "DataSource ID matches a system type or DataSource. Please choose another ID."
            }
        ]},
    //{name:"dataFormat", defaultValue:"iscServer", redrawOnChange:true},
    {type:"ComboBoxItem", name:"inheritsFrom", addUnknownValues:false,
        title:"Inherits From",
        showIf:"form.creator && form.creator.canEditInheritsFrom",
        
        valueMap:[],
        getValueMap : function () {
            return this.form.creator._getInheritsFromValueMap();
        },
        changed : function () {
            this.form.creator._inheritsFromUpdated(this.getValue());
        }
    },
    {type:"section", defaultValue:"XPath Binding", name:"xPathBindingSection",
     showIf:"form.showMainEditorSectionHeaders && (values.dataFormat != 'iscServer' && values.serverType != 'sql' && values.serverType != null && values._constructor != 'MockDataSource')",
     itemIds:["dataURL", "selectBy", "recordXPath", "recordName"]},
    {name:"dataURL", showIf:"values.dataFormat != 'iscServer' && values._constructor != 'MockDataSource'"},
    {name:"selectBy", title:"Select Records By", 
     shouldSaveValue:false,
     valueMap:{ tagName:"Tag Name", xpath:"XPath Expression" },
     defaultValue:"xpath",
     redrawOnChange:true,
     // can't use tagName in JSON
     showIf:"values.dataFormat == 'xml'"},
    // allowed in XML or JSON
    {name:"recordXPath", 
     showIf:"values.dataFormat != 'iscServer' && form.getItem('selectBy').getValue() == 'xpath' && values._constructor != 'MockDataSource'"},
    // allow in XML only
    {name:"recordName", 
     showIf:"values.dataFormat == 'xml' && values.selectBy == 'tagName'"},

    {type:"section", defaultValue:"SQL Binding", name:"sqlBindingSection",
     showIf:"form.showMainEditorSectionHeaders && !values.clientOnly && (values.serverType == 'sql' || values.serverType == 'hibernate')",
     itemIds:["dbName", "schema", "tableName"]},
    {name:"dbName", showIf:"!values.clientOnly && values.serverType == 'sql'", showHint: true, showHintInField: true, hint: "default"}, 
    {name:"schema", showIf:"!values.clientOnly && values.serverType == 'sql'", showHint: true, showHintInField: true, hint: "default"}, 
    {name:"tableName", 
     showIf:"!values.clientOnly && values.serverType == 'sql' || values.serverType == 'hibernate'",
     showHint: true, showHintInField: true, 
     hint: "same as ID"},

    {type:"section", defaultValue:"Record Titles", name:"recordTitlesSection", sectionExpanded:false,
        showIf:"form.showMainEditorSectionHeaders",
        itemIds:[/*"title", "pluralTitle",*/ "titleField"]},
    
    //{name:"title", showHint: true, showHintInField: true, hint: "same as ID"},
    //{name:"pluralTitle", showHint: true, showHintInField: true, hint: "same as ID + 's'"},
    {name:"titleField", editorType:"SelectItem", allowEmptyValue: true, valueMap: []},

    {type:"section", defaultValue:"Advanced", name:"advancedSection", sectionExpanded:false,
        showIf:"form.showMainEditorSectionHeaders && values.clientOnly != true && (values.serverType != null)",
        itemIds:["dropExtraFields", "autoDeriveSchema", "quoteTableName", "beanClassName"]},
    {name:"dropExtraFields", showIf:"values.clientOnly != true && values._constructor != 'MockDataSource' && values.serverType != 'sql'"},
    {name:"autoDeriveSchema", showIf:"values.clientOnly != true && values._constructor != 'MockDataSource'"},
    {name:"quoteTableName", showIf:"values.clientOnly != true && values.serverType == 'sql'"}, 
    {name:"beanClassName", 
     showIf:"values.clientOnly != true && values.serverType == 'sql' || values.serverType == 'hibernate'"}
],

_getInheritsFromValueMap : function () {
    var knownDataSources = this.knownDataSources || [];
    return knownDataSources.getProperty("ID");
},
_inheritsFromUpdated : function (parentDS) {
    // Re-bind defaults to FieldEditor
    if (this.showInheritedFields) this.bindFields(this.getDefaults());
},

fieldEditorDefaults: {
    _constructor: "ListEditor",
    autoDraw:false,
    inlineEdit:true,
    dataSource:"DataSourceField",
    saveLocally:true,
    minHeight:226,
    gridButtonsOrientation:"right",
    gridButtonsProperties: {
        height: "100%"
    },
    formProperties: { 
        numCols:4,
        colWidths:null,
        initialGroups:10
    },
    gridDefaults:{ 
        editEvent:"click",
        
        emptyMessage: "Loading DataSource definition... ${loadingImage}",

        listEndEditAction:"next",
        showNewRecordRow:true,
        newRecordRowMessage:"-- Click to add new field --",    
        canReorderRecords: true,

        autoParent:"gridLayout",
        selectionType:isc.Selection.SINGLE,
        recordClick:"this.creator.recordClick(record)",
        modalEditing:true,
        editorEnter:"this.creator.updateButtonStates()",
        cellChanged:"this.creator.updateButtonStates()",
        selectionUpdated: "this.creator.updateButtonStates()",
        contextMenu : {
            data : [
                {title:"Remove", click: "target.creator.removeRecord()" }
            ]
        },
        // get rid of default LG borders
        styleName:"rightBorderOnly",
        validateByCell:true,
        leaveScrollbarGap:false,
        alternateRecordStyles:true,
        // show a delete column
        canRemoveRecords:true,
        canEdit: true,
        canEditCell : function (rowNum, colNum) {
            var record = this.getRecord(rowNum),
                field = this.getField(colNum),
                fieldName = field.name,
                isNameOrTitle = (fieldName == "name" || fieldName == "title");
            // Cannot edit field that is inherited
            if (record && record._inheritedFrom) return false;
            // Cannot edit field that is a foreignKey directly
            if (record && record.foreignKey) return false;

            if (isc.isA.TreeGrid(this)) {
                if (record && record.isFolder &&
                    !(isNameOrTitle || fieldName == "required" || fieldName == "hidden"))
                {
                    return false;
                }
            }
            else {
                if (this.getDataSource().fieldIsComplexType(field) && !isNameOrTitle) 
                    return false;
            }
            // An includeFrom field only allows editing of the name and title
            if (!isNameOrTitle && record && record.includeFrom && fieldName != "hidden") {
                return false;
            }

            // Disallow editing PK if not allowed at all or row is being edited
            if (fieldName == "primaryKey" &&
                    (!this.creator.creator.canChangePrimaryKey || this.getEditRow() == rowNum))
            {
                return false;
            }
            return this.Super('canEditCell', arguments);
        },
        getCellCSSText : function (record, rowNum, colNum) {
            var fieldName = this.getFieldName(colNum),
                css = this.Super("getCellCSSText", arguments)
            ;
            // Show empty title field in hint style - auto-derived or related field details
            if (fieldName == "title" && record && !record.title && !this.isSelected(record)) {
                // hint color from Tahoe and Obsidian so it works in both dark and light skins
                css = "color:#999999;";
            }
            return css;
        },
        getEditorProperties : function (editField, editedRecord, rowNum) {
            var properties = this.Super("getEditorProperties", arguments);
            if (editField.name == "name" &&
                editedRecord && editedRecord.includeFrom && !editedRecord.name)
            {
                var name = editField._nameFromValueOrIncludeFrom(null, editedRecord.includeFrom),
                    hint = "Using related field name: <i>" + name + "</i>"
                ;
                isc.addProperties(properties, {
                    showHintInField: true,
                    hint: hint
                });
            } else if (editField.name == "title") {
                if (editedRecord && !editedRecord.title) {
                    var title,
                        hint
                    ;
                    if (editedRecord.includeFrom) {
                        var session = this.creator.creator.session;
                        title = editField._titleFromValueOrIncludeFrom(null, editedRecord.includeFrom, session);
                        hint = "<i>" + title + "</i>";
                    } else {
                        title = isc.DataSource.getAutoTitle(editedRecord.name);
                        hint = "<i>" + title + "</i>";
                    }
                    isc.addProperties(properties, {
                        showHintInField: true,
                        hint: hint
                    });
                }
            } else {
                // Clear hint which may be present from a previous record edit
                isc.addProperties(properties, {
                    hint: null
                });
            }
            return properties;
        },
        willAcceptDrop : function () {
            var recordNum = this.getEventRecordNum(null, true),
                dropRecord = this.data.get(recordNum);
            // Prevent drop within inherited fields
            if (dropRecord && dropRecord._inheritedFrom) return false;
            return this.Super("willAcceptDrop", arguments);
        },
        folderDrop : function (nodes, folder, index, sourceWidget) {
            if (this == sourceWidget) {
                // Reorder drop
                var editor = this.creator.creator,
                    dsName = editor.getDataSourceID(),
                    session = editor.session,
                    tree = this.data,
                    fieldList = tree.getNodeList(),
                    movedField = nodes[0],
                    movedFieldIndex = tree.indexOf(movedField),
                    movedBeforeField = (movedFieldIndex == 0 ? fieldList[1] : (fieldList[movedFieldIndex-1]._inheritedFrom ? fieldList[movedFieldIndex+1] : null)),
                    movedAfterField = (!movedBeforeField ? fieldList[movedFieldIndex-1] : null), 
                    targetNodes = tree.getChildren(folder),
                    targetBeforeField = targetNodes && targetNodes[index],
                    targetAfterField = targetNodes && (!targetBeforeField ? targetNodes[index - 1] : null)
                ;
                session.addChange(dsName, "reorderField", {
                    fieldName: movedField.name,
                    fromPosition: (movedBeforeField ? "before" : "after"),
                    fromField: (movedBeforeField || movedAfterField).name,
                    toPosition: (targetBeforeField ? "before" : "after"),
                    toField: (targetBeforeField || targetAfterField).name
                });
            }
            this.Super("folderDrop", arguments);
        },
        editComplete : function (rowNum, colNum, newValues, oldValues, editCompletionEvent) {
            var undef;
            if (newValues._removing) {
                // Removing a field is already handled in _deleteField()
                return;
            }
            if ((oldValues && oldValues._newField) ||
                ((!oldValues || !oldValues.name) && newValues.name != undef))
            {
                var record = this.getRecord(rowNum);
                if (record) {
                    delete record._newField;
                    this.creator._fieldAdded(record);
                    // Just like LG.selectOnEdit
                    this.selectSingleRecord(rowNum);
                }
            } else {
                var record = this.getRecord(rowNum);
                this.creator._fieldChanged(record, oldValues, newValues);
            }
        },
        editorEnter : function (record, value, rowNum, colNum) {
            if (record && record._newField && this.getEditedCell(rowNum, "type") == null) {
                // Default new fields to type "text".
                // This also sets the record to dirty so saving without making other
                // changes still fires the fieldAdded() event.
                this.setEditValue(rowNum, "type", "text");
            }
            if (record && record._newField && this.getEditedCell(rowNum, "name") == null) {
                // Default new fields to name "" which forces a manual entry and correct
                // validation.
                this.setEditValue(rowNum, "name", "");
            }
        },
        removeRecordClick : function (rowNum) {
            var grid = this,
                fieldEditor = grid.creator,
                record = this.getRecord(rowNum)
            ;
            // if there's no record, nothing to do
            if (!record) return;

            // If PK is potentially an autoAdd, don't allow the PK to be removed
            if (this.creator.creator.autoAddPK && record.primaryKey) {
                isc.Hover.show("Primary key field cannot be removed");
                return;
            }
            
            if (record.foreignKey) {
                var currentDSName = grid.creator.creator.getDataSourceID(),
                    relatedDSName = isc.DS.getForeignDSName(record, grid.creator.dataSource)
                ;
                this.showRemoveRelationDialog(currentDSName, relatedDSName, function (removeField) {
                    if (removeField == true) {
                        // Remove relation and relation field (FK)
                        fieldEditor._deleteField(record);
                    } else {
                        // Remove relation but not the relation field (FK)
                        // Remove affected includeFrom fields for the relation
                        fieldEditor._deleteFieldRelation(record);
                    }
                }.bind(this));

                // TODO: If other project DataSources reference this field as either
                // a foreignKey or includeFrom, these related DataSources must be updated.
                // Should there be a user warning?

            } else {
                fieldEditor._deleteField(record);
            }
        },
        showRemoveRelationDialog : function (currentDSName, relatedDSName, callback) {
            var message = "Removing this relationship will remove all existing links between " +
                "'${currentDS}' records and '${relatedDS}' records if you save."
            ;
            message = message.evalDynamicString(this, {
                currentDS: currentDSName,
                relatedDS: relatedDSName
            });

            var header = isc.Label.create({
                width: "100%",
                height: 30,
                contents: message
            });
        
            var valueMap = {
                false: "Remove relation; keep relation data",
                true: "Remove relation and remove DataSource field holding relation data"
            };
        
            var choiceForm = isc.DynamicForm.create({
                width: "100%",
                height: "*",
                numCols: 1,
                items: [
                    { name: "removeField", showTitle: false, type: "boolean",
                        editorType: "radioGroup",
                        width: "*", wrap: false,
                        defaultValue: false,
                        valueMap: valueMap
                    }
                ]
            });
        
            var cancelButton = isc.IButton.create({
                autoDraw: false,
                title: "Cancel",
                width: 75,
                click : function () {
                    this.topElement.closeClick();
                }
            });
            var okButton = isc.IButton.create({
                autoDraw: false,
                title: "OK",
                width: 75,
                click : function () {
                    var removeField = choiceForm.getValue("removeField");
                    this.topElement.closeClick();
                    callback(removeField);
                }
            });
            var buttonsLayout = isc.HLayout.create({
                autoDraw: false,
                height: 30,
                membersMargin: 10
            });
            buttonsLayout.addMembers([ isc.LayoutSpacer.create(), cancelButton, okButton ]);
        
            var layout = isc.VLayout.create({
                width: "100%",
                height: "100%",
                padding: 10,
                membersMargin: 10,
                members: [ header, choiceForm, buttonsLayout ]
            });
        
            var dialog = isc.Window.create({
                ID: "removeRelationDialog",
                title: "Remove relation?",
                width: 550,
                height: 250,
                isModal: true,
                showModalMask: true,
                autoCenter: true,
                items: [ layout ],
                close : function () {
                    this.Super("close", arguments);
                    this.markForDestroy();
                }
            });
        
            dialog.show();
        },
        // When tabbing into a new record, trigger adding a new record so that the edit
        // has the needed linkage to parent in tree
        rowEditorEnter : function (record, editValues, rowNum) {
            if (record || !this.creator.creator.canEditChildSchema) {
                return;
            }

            this.delayCall("_handleEditNewRecord");
            return false;
        },
        _handleEditNewRecord : function () {
            this.cancelEditing();
            this.creator.newRecord();
        },
        recordClick : function (viewer,record,recordNum,field,fieldNum,value,rawValue) {
            if (recordNum == this.getTotalRows()-1) {
                // Click on "add new record" row in grid
                viewer.creator.newRecord();
                return false;
            }
            return this.Super("recordClick", arguments);
        },
        recordDoubleClick : function (viewer, record, recordNum, field, fieldNum, value, rawValue) {
            var dsEditor = viewer.creator.creator;
            if (record.foreignKey && dsEditor.enableRelationEditor) {
                dsEditor.editRelations(record.foreignKey);
                return false;
            }
        },

        _$linkTemplate:[
            "<a href='",
            ,   // 1: HREF
            "' target='",
            ,   // 3: name of target window
            // onclick handler enables us to prevent popping a window if (EG) we're masked.
            //                      5: ID
            "' onclick='if(window.",     ,") return ",
                    //  7:ID                               9:dsName,
                             ,"._linkToDataSourceClicked(\"",        ,"\");'>",
            ,   // 11: link text
            "</a>"
        ],

        createDSLink : function (dsName) {
            var dsEditor = this.creator.creator;
            if (!dsEditor.canNavigateToDataSource) {
                return dsName;
            }

            var ID = this.getID(),
                template = this._$linkTemplate
            ;
            template[1] = "javascript:void";
            template[3] = "javascript";
            template[5] = ID;
            template[7] = ID;
            template[9] = dsName;
            template[11] = dsName;

            return template.join(isc.emptyString);
        },

        // Format a field of format [[<dsName>].<dsName>.]<fieldName> so that the
        // <fieldName> DataSource is a link for quick navigation
        formatRelatedField : function (name) {
            var split = name.split(".");
            if (!split || split.length < 2) {
                return name;
            }

            var dsNames = split.slice(0, split.length-1),
                fieldName = split[split.length-1],
                self = this,
                nameParts = dsNames.map(function (dsName) {
                    return self.createDSLink(dsName);
                })
            ;
            nameParts.add(fieldName);

            return nameParts.join(".");
        },

        _linkToDataSourceClicked : function (dsName) {
            var dsEditor = this.creator.creator;
            // If already editing the DS, nothing else to do
            if (dsEditor._currentDSName == dsName) {
                return false;
            }

            // Open clicked DataSource
            dsEditor.openDataSource(dsName);

            // Called from an <a href .../> onclick so return false to cancel action
            return false;
        }
    },

    _fieldChanged : function (record, oldValues, newValues) {
        var _this = this,
            dsEditor = this.creator,
            session = dsEditor.session,
            dsName = dsEditor.getCurrentDataSourceID(),
            undef;
        newValues = record;
        var nameChanged = (oldValues && oldValues.name != null &&
            newValues.name != undef &&
            oldValues.name != newValues.name);
        var typeChanged = (oldValues && oldValues.type != null &&
            newValues.type != undef &&
            oldValues.type != newValues.type);
        var lengthChanged = (oldValues && 
            ((oldValues.length == null && newValues.length != undef) ||
                (oldValues.length != null &&
                    newValues.length != undef &&
                    oldValues.length != newValues.length)));

        var applyChanges = function () {
            var fieldName = oldValues.name;
            if (nameChanged) {
                _this._fieldNameChanged(oldValues.name, newValues.name);
                fieldName = newValues.name;
            }
            if (typeChanged || lengthChanged) {
                _this._fieldTypeChanged(record,
                                        (typeChanged ? newValues.type : oldValues.type),
                                        (lengthChanged ? newValues.length : oldValues.length));
            }
            // See what other properties have changed

            var excludeKeys = ["name","type","length","id","parentId"],
                keys = isc.getKeys(oldValues).addList(isc.getKeys(newValues)).getUniqueItems().sort(),
                changeContext;
            for (var i = 0; i < keys.length; i++) {
                var key = keys[i];
                if (key.startsWith("_") || excludeKeys.contains(key)) continue;

                var oldValue = oldValues[key],
                    newValue = newValues[key];
                if (oldValue != newValue) {
                    var context = session.addChange(dsName, "changeFieldProperty", {
                        fieldName: fieldName,
                        property: key,
                        originalValue: oldValues[key],
                        newValue: newValues[key]
                    });
                    if (!changeContext) {
                        changeContext = context;
                        session.startChangeContext(changeContext);
                    }
                }
            }
            if (changeContext) session.endChangeContext(changeContext);
        };

        if (this._skipTypeChangeProcessing) typeChanged = false;

        // If type has changed and the field has validators, ask user if the validators
        // should be dropped
        if (typeChanged && record.validators && record.validators.length > 0) {
            isc.confirm("Changing the field type may cause some validators to be invalid. Clear field validators?", function (value) {
                if (value) {
                    delete record.validators;
                }
                applyChanges();
            });
            return;
        }

        // All other changes can be immediately applied
        applyChanges();
    },

    _fieldNameChanged : function (fromName, toName) {
        // Previous field name may have been used in validator.applyWhen values - update these
        var dsEditor = this.creator,
            session = dsEditor.session,
            dsName = dsEditor.getCurrentDataSourceID()
        ;

        // Remove the field from the session defaults and grab the updated defaults
        var defaults = session.renameField(dsName, fromName, toName);

        // Rebind the field editor with the updated fields
        dsEditor.bindFields(defaults);
        
        if (dsEditor.editSampleData) {
            // Re-create testDS with updated fields/data
            dsEditor._rebindSampleDataGrid(defaults.fields, defaults.cacheData);
        }

        // Update selections and possibly value of titleField. Must come after sampleData updated
        dsEditor.updateTitleField(fromName, toName);

        this.fieldNameChanged(fromName, toName);
    },
    fieldNameChanged : function (fromName, toName) {
    },
    _fieldTypeChanged : function (field, newType, newLength) {
        if (this._skipTypeChangeProcessing) return;

        var dsEditor = this.creator,
            session = dsEditor.session,
            dsName = dsEditor.getCurrentDataSourceID()
        ;

        // Update field type in session
        var defaults = session.changeFieldType(dsName, field.name, newType, newLength);

        // Rebind the field editor with the updated fields
        dsEditor.bindFields(defaults);

        if (dsEditor.editSampleData) {
            // Re-create testDS with updated fields/data
            dsEditor._rebindSampleDataGrid(defaults.fields, defaults.cacheData);
        }

        // Update selections and possibly value of titleField. Must come after sampleData updated
        dsEditor.updateTitleField();

        this.fieldTypeChanged(field);
    },
    fieldTypeChanged : function (field) {
    },
    _fieldAdded : function (field) {
        var dsEditor = this.creator,
            session = dsEditor.session
        ;

        var grid = this.grid,
            tree = grid.data
        ;
        field = (isc.isA.Tree(tree) ? tree.getCleanNodeData(field, false) : field);

        // Update DSE session with change
        var defaults = dsEditor.getDefaults();
        session.set(defaults.ID, defaults);

        // Save field add
        session.recordAddedField(defaults.ID, field);

        // Update Sample Data grid
        this.creator.rebindSampleData();

        // Update selections and possibly value of titleField. Must come after sampleData updated
        this.creator.updateTitleField();

        this.updateIncludeFieldButtonState();

        this.fieldAdded(field);
    },
    fieldAdded : function (field) {
    },
    _deleteField : function (field) {
        var dsEditor = this.creator,
            session = dsEditor.session
        ;

        // Previous field name may have been used in validator.applyWhen values.
        // Let user know these issues remain.
        var grid = this.grid,
            tree = grid.data,
            allFields = (isc.isA.Tree(tree) ? tree.getAllNodes() : tree),
            fieldName = field.name,
            referenced = false
        ;
        for (var i = 0; i < allFields.length; i++) {
            var f = allFields[i];
            if (f.validators && f.validators.length > 0) {
                for (var j = 0; j < f.validators.length; j++) {
                    referenced = this.validatorReferencesField(f.validators[j], fieldName) || referenced;
                }
            }
        }
        if (referenced) {
            isc.warn("Deletion of field " + fieldName + " affects one or more other fields " +
                "with validators that referenced this field. These affected criterion will be ignored.");
        }

        // Update DSE session with any changes
        var defaults = dsEditor.getDefaults();
        session.set(defaults.ID, defaults);

        // Remove the field from the session defaults and grab the updated defaults
        defaults = session.removeField(defaults.ID, field.name, field.foreignKey);

        // Rebind the field editor with the updated fields
        dsEditor.bindFields(defaults);

        if (dsEditor.editSampleData) {
            // Re-create testDS with updated fields/data
            dsEditor._rebindSampleDataGrid(defaults.fields, defaults.cacheData);
        }

        // Update selections and possibly value of titleField. Must come after sampleData updated
        dsEditor.updateTitleField();

        this.delayCall("updateButtonStates");
        this.updateIncludeFieldButtonState();

        this.fieldDeleted(field);
    },
    fieldDeleted : function (field) {
    },
    // Remove relation from field along with associated includes. Does not remove FK field.
    _deleteFieldRelation : function (field) {
        var dsEditor = this.creator,
            session = dsEditor.session
        ;

        // Previous field name may have been used in validator.applyWhen values.
        // Let user know these issues remain.
        var grid = this.grid,
            tree = grid.data,
            allFields = (isc.isA.Tree(tree) ? tree.getAllNodes() : tree),
            fieldName = field.name,
            referenced = false
        ;
        for (var i = 0; i < allFields.length; i++) {
            var f = allFields[i];
            if (f.validators && f.validators.length > 0) {
                for (var j = 0; j < f.validators.length; j++) {
                    referenced = this.validatorReferencesField(f.validators[j], fieldName) || referenced;
                }
            }
        }
        if (referenced) {
            isc.warn("Deletion of field " + fieldName + " affects one or more other fields " +
                "with validators that referenced this field. These affected criterion will be ignored.");
        }

        // Update DSE session with any changes
        var defaults = dsEditor.getDefaults();
        session.set(defaults.ID, defaults);

        var defaultsField = defaults.fields.find("name", field.name),
            foreignKey = defaultsField.foreignKey
        ;

        // Remove relation but not the relation field (FK)
        var changeContext = session.addChange(defaults.ID, "changeFieldProperty", {
            fieldName: field.name,
            property: "foreignKey",
            originalValue: field.foreignKey,
            newValue: null
        });
        delete defaultsField.foreignKey;
        session.startChangeContext(changeContext);

        if (defaultsField.foreignDisplayField) {
            session.addChange(defaults.ID, "changeFieldProperty", {
                fieldName: field.name,
                property: "foreignDisplayField",
                originalValue: defaultsField.foreignDisplayField,
                newValue: null
            });
            delete defaultsField.foreignDisplayField;
        }
        if (defaultsField.useLocalDisplayFieldValue) {
            session.addChange(defaults.ID, "changeFieldProperty", {
                fieldName: field.name,
                property: "useLocalDisplayFieldValue",
                originalValue: defaultsField.useLocalDisplayFieldValue,
                newValue: null
            });
            delete defaultsField.useLocalDisplayFieldValue;
        }
        if (defaultsField.joinType) {
            session.addChange(defaults.ID, "changeFieldProperty", {
                fieldName: field.name,
                property: "joinType",
                originalValue: defaultsField.joinType,
                newValue: null
            });
            delete defaultsField.joinType;
        }
        if (defaultsField.displayField && defaults.fields.find("name", defaultsField.displayField) == null) {
            session.addChange(defaults.ID, "changeFieldProperty", {
                fieldName: field.name,
                property: "displayField",
                originalValue: defaultsField.displayField,
                newValue: null
            });
            delete defaultsField.displayField;
        }

        // Remove affected includeFrom fields for the relation
        session.removeIncludeFieldsForForeignKey(defaults.ID, foreignKey);

        session.endChangeContext(changeContext);

        // Rebind the field editor with the updated fields
        dsEditor.bindFields(defaults);

        // Update selections and possibly value of titleField. Must come after sampleData updated
        dsEditor.updateTitleField();

        this.delayCall("updateButtonStates");
        this.updateIncludeFieldButtonState();
    },

    updateButtonStates : function (currentRecord) {
        this.Super("updateButtonStates", arguments);
        var selectedRecord = this.grid.getSelectedRecord(),
            selectedRowNum = (selectedRecord ? this.grid.getRecordIndex(selectedRecord) : null),
            validSelection = selectedRecord && !this.grid.rowHasErrors(selectedRowNum),
            fieldType = selectedRecord && this.creator.getFieldType(selectedRecord),
            record = currentRecord || selectedRecord,
            haveFields = (this && this.grid && this.grid.data)
        ;

        this.setFieldButtonState(this.creator.derivedValueButton,
            (!validSelection || record._inheritedFrom || record.includeFrom || record.primaryKey || record.foreignKey || record.type == "sequence"),
            record && this.fieldHasFormula(record));
        this.setFieldButtonState(this.creator.validatorsButton,
            (!validSelection || record._inheritedFrom),
            record && this.fieldHasValidators(record));
        this.setFieldButtonState(this.creator.legalValuesButton,
            (!validSelection || record._inheritedFrom || record.type != "enum"),
            record && record.valueMap);
        this.setFieldButtonState(this.creator.formattingButton,
            (!validSelection || record._inheritedFrom || 
                (fieldType != "integer" && fieldType != "float" &&
                    fieldType != "date" && fieldType != "datetime" &&
                    fieldType != "time")),
            record && record.format);
        this.setFieldButtonState(this.creator.securityButton,
            (!validSelection || record._inheritedFrom),
            record && this.fieldHasSecurity(record));
        this.setFieldButtonState(this.creator.relationsButton,
            !haveFields,
            false);
    },

    updateIncludeFieldButtonState : function () {
        var editor = this.creator,
            session = editor.session,
            dsRelations = session.getRelations(),
            haveFields = (this.grid && this.grid.data)
        ;
        if (!dsRelations || !this.creator.includeFieldButton) return;

        var dsId = this.creator.getDataSourceID(),
            relations = dsRelations.getRelationsForDataSource(dsId)
        ;
        this.creator.includeFieldButton.setDisabled(!relations || relations.length == 0 || !haveFields);
    },

    setFieldButtonState : function (button, disabled, hasValue) {
        if (!button) return;
        if (disabled) {
            button.disable();
            if (button.icon) {
                button.setIcon(null);
            }
        } else {
            button.enable();
            if (hasValue && !button.icon) {
                if (button.iconSize != 10) button.iconSize = 10;
                button.setIcon("[SKINIMG]actions/accept.png");
            } else if (!hasValue && button.icon) {
                button.setIcon(null);
            }
        }
    },

    fieldHasFormula : function (record) {
        return (record.formula || record.template || record.valueFrom);
    },

    fieldHasValidators : function (record) {
        var validators = record.validators,
            result = (validators && validators.length > 0)
        ;  
        if (result) {
            // Don't include generated validators in result. These wouldn't be shown
            // in the validator editor.
            validators = validators.filter(function(v) {
                return !v.typeCastValidator && !v._generated && !v._typeValidator;
            });
            result = (validators.length > 0);
        }
        return result;
    },

    fieldHasSecurity : function (record) {
        return (record.viewRequiresRole != null ||
                record.viewRequires != null ||
                record.editRequiresRole != null ||
                record.editRequires != null);
    },

    updateValidatorFieldNames : function (validator, fromName, toName) {
        var applyWhen = validator.applyWhen;
        if (!applyWhen || isc.isA.emptyObject(applyWhen)) return;
        this._replaceCriteriaFieldName(applyWhen, fromName, toName);
    },

    _replaceCriteriaFieldName : function (criteria, fromName, toName) {
        var operator = criteria.operator,
            changed = false
        ;
        if (operator == "and" || operator == "or") {
            var innerCriteria = criteria.criteria;
            for (var i = 0; i < innerCriteria.length; i++) {
                if (this._replaceCriteriaFieldName(innerCriteria[i], fromName, toName)) {
                    changed = true;
                }
            }
        } else {
            if (criteria.fieldName != null && criteria.fieldName == fromName) {
                criteria.fieldName = toName;
                changed = true;
            }
        }
        return changed;
    },

    validatorReferencesField : function (validator, fieldName) {
        var applyWhen = validator.applyWhen;
        if (!applyWhen || isc.isA.emptyObject(applyWhen)) return false;
        return this._criteriaHasMatchingFieldName(applyWhen, fieldName);
    },

    _criteriaHasMatchingFieldName : function (criteria, fieldName) {
        var operator = criteria.operator;
        if (operator == "and" || operator == "or") {
            var innerCriteria = criteria.criteria;
            for (var i = 0; i < innerCriteria.length; i++) {
                if (this._criteriaHasMatchingFieldName(innerCriteria[i], fieldName)) {
                    return true;
                }
            }
        } else {
            if (criteria.fieldName != null && criteria.fieldName == fieldName) {
                return true;
            }
        }
        return false;
    },

    // override
    newRecord : function (defaultValues) {
        if (this.creator.canEditChildSchema) {
            var grid = this.grid,
                tree = grid.data,
                selectedNode = this.getSelectedNode();
                
            if (!selectedNode) selectedNode = tree.root;
            var parentNode = tree.getParent(selectedNode)

            if (selectedNode) {
                if (!tree.isFolder(selectedNode)) selectedNode = parentNode;
                var id = this.getNextUnusedNodeId();
                if (defaultValues) {
                    var newNode = isc.addProperties({}, defaultValues, {
                        id: id,
                        parentId: selectedNode ? selectedNode.id : null
                    });
                    this.addNode(newNode, selectedNode);
                    this._fieldAdded(newNode);
                } else {
                    var newNode = isc.addProperties({ 
                            name: this.getNextUniqueFieldName(selectedNode, "field"),
                            id: id,
                            parentId: selectedNode ? selectedNode.id : null,
                            _newField: true
                        });
                    this.addNode(newNode, selectedNode);
                    var node = grid.findByKey(id);
                    if (node) {
                        var rowNum = grid.getRecordIndex(node);
                        if (rowNum >= 0) {
                            grid.startEditing(rowNum);
                        }
                    }
                }
            }
        } else this.Super("newRecord", arguments);
    },
    saveRecord : function () {
        if (!this.form.validate()) return false;
        var newValues = this.form.getValues();  
    
        // Record changes
        if (this.form.saveOperationType == "add") {
            this._fieldAdded(newValues);
        } else {
            var record = this.getEditRecord(),
                tree = this.grid.data,
                oldValues = (isc.isA.Tree(tree) ? tree.getCleanNodeData(record, true) : record) || {}
            ;
            this.creator.fieldEditor._fieldChanged(record, oldValues, newValues);
        }
        // Actually save values into field
        return this.Super("saveRecord", arguments);
    },
    getSelectedNode : function () {
        return this.grid.getSelectedRecord();
    },
    addNode : function (newNode, parentNode) {
        var tree = this.grid.data,
            position
        ;
        // If a new node has an includeFrom value it must have been created by selecting
        // an include field. These fields should be added immediately after the corresponding
        // foreignKey but after any other includeFrom fields for the same DataSource.
        if (newNode.includeFrom) {
            var split = newNode.includeFrom.split(".");
            if (split && split.length >= 2) {
                // Find relation node
                var dsName = split[split.length-2],
                    nodes = tree.getAllNodes(),
                    relationPosition
                ;
                for (var i = 0; i < nodes.length; i++) {
                    var node = nodes[i];
                    if (node.foreignKey) {
                        split = node.foreignKey.split(".");
                        if (split && split.length >= 2) {
                            var foreignKeyDSName = split[split.length-2];
                            if (foreignKeyDSName == dsName) {
                                relationPosition = i;
                                break;
                            }
                        }
                    }
                }
                if (relationPosition != null && relationPosition < nodes.length-1) {
                    // Find next node position under relation after any other includeFrom
                    // fields against the same DataSource
                    for (var i = relationPosition+1; i < nodes.length; i++) {
                        var node = nodes[i];
                        if (!node.includeFrom) {
                            // no more includeFroms to check
                            position = i;
                            break;
                        }

                        split = node.includeFrom.split(".");
                        if (!split || split.length < 2) {
                            // includeFrom points to a field on this DS; put new node here
                            position = i;
                            break;
                        }
                        // Find relation node
                        var nodeDSName = split[split.length-2];
                        if (dsName != nodeDSName) {
                            // found another includeFrom but it's against another DataSource
                            position = i;
                            break;
                        }
                        // current node must be an includeFrom against the target DS; keep looking
                    }
                }
            }
        }
        tree.add(newNode, parentNode, position);
    },
    getNextUniqueFieldName : function (node, prefix) {
        // Only suppress assigning a field name to "field" prefix values so
        // that "child" fields will have an assigned name because those need
        // to be manually linked into the tree.
        if (prefix == "field" && this.autoAssignNewFieldName == false) return null;

        var childFields = node ? node.fields || [] : [],
	        inc=1;

        if (!prefix || prefix.length == 0) prefix = "field";
        if (childFields && childFields.length > 0) {
            for (var i = 0; i < childFields.length; i++) {
                var item = childFields.get(i), 
                    itemName = item.name;
                // An includeFrom field doesn't need an explicit name
                if (!itemName && item.includeFrom) {
                    itemName = item.includeFrom;
                    var dotIndex = itemName.lastIndexOf(".");
                    if (dotIndex >= 0) itemName = itemName.substring(dotIndex + 1);
                }
                if (itemName.substring(0, prefix.length) == prefix && itemName.length > prefix.length) {
                    var thisInc = parseInt(itemName.substring(prefix.length));
                    if (!isNaN(thisInc) && thisInc >= inc) 
                        inc = thisInc+1;
                }
            }
        }
        return prefix + inc;
    },
    getNextUnusedNodeId : function () {
        var tree = this.grid.data;
        for (var i = 1; i<10000; i++) {
            var item = tree.findById(i);
            if (!item) return i;
        }
        return 1;
    },

    baseFieldTypesValueMap: {
        "text": "text: a normal text value",
        "enum": "enum: a text field allowing only certain values",
        "integer": "integer: a whole number",
        "float": "float: a fractional or decimal number",
        "boolean": "boolean: only a true or false allowed",
        "date": "date: a specific date, with no time",
        "time": "time: a specific time, with no date",
        "datetime": "datetime: a specific time on a specific date",
        "sequence": "sequence: an automatically managed numeric unique identifier for records",
        "URL": "URL: a link to something on the web",
        "image": "image: a link to an image on the web",
        "color": "color: a color value",
        "phoneNumber": "phoneNumber: a phone number",
        "creatorTimestamp": "creatorTimestamp: date/time of record add",
        "modifierTimestamp": "modifierTimestamp: data/time of record modifications"
    },
    binaryFieldTypesValueMap: {
        "imageFile": "imageFile: an image stored in this DataSource",
        "binary": "binary: any binary file that is not an image"
    },

    getFieldTypeValueMap : function () {
        var valueMap,
            isMockDataSource = (isc.isA.MockDataSource(this.targetDataSource) ||
                (isc.isAn.Object(this.targetDataSource) && this.targetDataSource._constructor == "MockDataSource"))
        ;
        if (!isMockDataSource && !this.targetDataSource.clientOnly) {
            valueMap = this._binaryFieldTypesValueMap;
            if (!valueMap) {
                valueMap = this._binaryFieldTypesValueMap = isc.addProperties({}, this.baseFieldTypesValueMap, this.binaryFieldTypesValueMap);
            }
        } else {
            valueMap = this.baseFieldTypesValueMap;
        }
        return valueMap;
    },
    // Default ListEditor doesn't validate the grid
    validate : function () { 
        return this.Super("validate", arguments) && !this.grid.hasErrors();
    },

    // Utilities to be shared amoung grid field event handlers
    confirmRemovingRelations : function (targetFieldName, callback) {
        var dsEditor = this.creator,
            grid = this.grid,
            dsRelations = dsEditor.dsRelations
        ;
        if (!grid._originalPKFieldName && dsRelations) {
            var dsId = dsEditor.getDataSourceID(),
                relations = dsRelations.getRelationsForDataSource(dsId),
                relatedDataSources = []
            ;
            if (isc.isAn.Array(relations)) {
                var currentPkField = grid.data.find("primaryKey", true);
                for (var i = 0; i < relations.length; i++) {
                    var relation = relations[i];
                    if (relation.relatedFieldName == currentPkField.name) {
                        relatedDataSources.add(relation.dsId);
                    }
                }
            }
            if (relatedDataSources.length > 0) {
                var names = (relatedDataSources.length > 2 ?
                    relatedDataSources.slice(0,-1).join(",") + " and " + relatedDataSources.slice(-1) :
                    (relatedDataSources.length > 1 ?
                        relatedDataSources.join(" and ") :
                        relatedDataSources[0])),
                    plural = (relatedDataSources.length > 1 ? "s" : ""),
                    has = (relatedDataSources.length > 1 ? "have" : "has a"),
                    depends = (relatedDataSources.length > 1 ? "depend" : "depends")
                ;
                isc.confirm("DataSource" + plural + " <i>" + names + "</i> " + has +
                    " relation" + plural + " that " + depends + " on the current" +
                    " primary key of this DataSource. Changing the type of <i>" +
                    targetFieldName + "</i> to a sequence also moves the primary" +
                    "key.<P>" +
                    "Are you sure you want to change the type and primary key? The " +
                    "relation will be removed.",
                    function (response)
                {
                    // OK means remove relations; cancel ignores click
                    if (response) {
                        callback();
                    }
                }, {
                    buttons: [
                        isc.Dialog.CANCEL,
                        { title: "Remove Relation" + plural,
                            width:150, overflow: "visible",
                            click: function () { this.topElement.okClick(); }
                        }
                    ],
                    autoFocusButton: 0
                });

                return;
            }
        }
        // No relation to remove
        this.fireCallback(callback);
    },

    changePrimaryKey : function (rowNum, targetField, action, callback) {
        // action = "keepAsSequence" | "newSequence" | "keepValues"
        var dsEditor = this.creator,
            grid = this.grid
        ;

        // Clear previous primaryKey selection
        var currentPkField = grid.data.find("primaryKey", true),
            currentPkRowNum = grid.getRecordIndex(currentPkField)
        ;
        grid.startEditing(currentPkRowNum);
        if (currentPkRowNum != rowNum) {
            grid.setEditValue(currentPkRowNum, "primaryKey", false);
        }

        // Set flag to prevent normal field type processing from occurring. The
        // type changes that may occur here are fully handled here.
        this._skipTypeChangeProcessing = true;

        // If new primaryKey will be converted to a sequence and the current primaryKey field
        // is a sequence we must reset the old one so there is only one sequence field
        if (action == "newSequence" &&
            targetField.type != "sequence" &&
            grid.getEditedRecord(currentPkRowNum).type == "sequence" &&
            currentPkRowNum != rowNum)
        {
            grid.setEditValue(currentPkRowNum, "type", "integer");
        }

        // Save off the original primaryKey fieldName so we know whether
        // relations should be removed. They should if the PK has changed
        // and there are relations to this DS.
        if (!grid._originalPKFieldName) {
            grid._originalPKFieldName = currentPkField.name;
        }

        // If setting primary key back to the original field, we don't need
        // the original PK field name anymore - it hasn't changed.
        if (targetField.name == grid._originalPKFieldName) {
            delete grid._originalPKFieldName;
        }
        grid.endEditing();

        // Set new primaryKey selection and update data as needed
        grid.startEditing(rowNum);
        grid.setEditValue(rowNum, "primaryKey", true);

        if (action == "keepAsSequence") {
            // If the new PK field is already a sequence, don't force it hidden
            var changedType = false;
            if (targetField.type != "sequence") {
                grid.setEditValue(rowNum, "type", "sequence");
                grid.setEditValue(rowNum, "hidden", true);
                changedType = true;
            }
            grid.endEditing();
            if (changedType) {
                var fields = dsEditor.getFields(),
                    records = dsEditor.getSampleData() || []
                ;
                dsEditor.__rebindSampleDataGrid(fields, records);
            }
        } else if (action == "newSequence") {
            // If the new PK field is already a sequence, don't force it hidden
            var changedType = false;
            if (targetField.type != "sequence") {
                grid.setEditValue(rowNum, "type", "sequence");
                grid.setEditValue(rowNum, "hidden", true);
                grid.setEditValue(rowNum, "length", null);
                grid.setEditValue(rowNum, "valueMap", null);
                changedType = true;
            }
            grid.endEditing();

            if (dsEditor.editSampleData) {
                var fields = dsEditor.getFields(),
                    records = dsEditor.getSampleData() || []
                ;

                if (changedType) {
                    // Since type also changed for the primaryKey, reassign sequence
                    // and update sample data grid
                    if (records.length > 0) {
                        for (var i = 0; i < records.length; i++) {
                            records[i][targetField.name] = i+1;
                        }
                    }
                    dsEditor.__rebindSampleDataGrid(fields, records);
                } else {
                    // Assign updated sequence in sample data grid
                    if (records.length > 0) {
                        var sampleDataGrid = dsEditor.dataGrid,
                            rowCount = sampleDataGrid.getTotalRows()
                        ;
                        for (var i = 0; i < rowCount; i++) {
                            sampleDataGrid.setEditValue(i, targetField.name, rowNum);
                        }
                        sampleDataGrid.markForRedraw("created new sequence");
                    }
                }
            }
        } else {
            grid.setEditValue(rowNum, "required", true);
            // Don't allow field value to be updated - only entered for new records
            targetField.canUpdate = false;
            grid.endEditing();
        }

        delete this._skipTypeChangeProcessing;

        this.fireCallback(callback);
    },

    // override
    editMore : function () {
        this.Super("editMore", arguments);
        // Enforce non-editibility of inherited fields
        var currentRecord=this.getEditRecord();
        this.form.setCanEdit(currentRecord._inheritedFrom == null);
    }
},

//>@attr dataSourceEditor.canEditMultipleAttribute (boolean : true : IR)
// Should this DataSourceEditor allow users to set +link{dataSourceField.multiple}
// @visibility reifyOnSite
//<
canEditMultipleAttribute:true,

//>@attr dataSourceEditor.canEditInheritsFrom (boolean : false : IR)
// Should this DataSourceEditor allow users to specify a parent dataSource by setting 
// +link{dataSource.inheritsFrom}.
// <P>
// If true, the user will be able to pick a parent dataSource from the current set of
// +link{knownDataSources}.
// @visibility reifyOnSite
//<

canEditInheritsFrom:false,




getFieldEditorGridFields : function () {
    return isc.clone(this.fieldEditorGridFields);
},
fieldEditorGridFields:[
    {name:"title", treeField: true, width: "*",
        prompt: "The name of this field that users of your applications will see",
        // Where includeFrom has been used, the title defaults to includeFrom's title.
        // Show that instead of nothing. Same for a standard field with no explicit title.
        formatCellValue : function(value, record, rowNum, colNum, grid) {
            if (!record) record = {};
            var session = grid.creator.creator.session,
                formattedValue = this._titleFromValueOrIncludeFrom(value, record.includeFrom, session)
            ;
            if (record.includeFrom && !record.title) {
                formattedValue = "<i>" + formattedValue + "</i>";
            } else if (!record.title) {
                var title = isc.DataSource.getAutoTitle(record.name);
                formattedValue = "<i>" + title + "</i>";
            }
            return formattedValue;
        },

        showHover: true,
        hoverWrap: false,
        hoverHTML : function (record, value, rowNum, colNum, grid) {
            var hover;
            if (record.includeFrom && !record.title) {
                hover = "Using title from related field";
            } else if (!record.title) {
                hover = "Title automatically derived from field name";
            }
            return hover;
        },

        // If the value is present, return it. Otherwise, return the last
        // part of the includeFrom -- which is what the name defaults to.
        _titleFromValueOrIncludeFrom : function(value, includeFrom, session) {

            if (value || !includeFrom) {
                return value;
            } else {
                if (includeFrom) {
                    var split = includeFrom.split(".");
                    if (split && split.length >= 2) {
                        var dsName = split[split.length-2],
                            dsField = split[split.length-1],
                            ds = session.get(dsName)
                        ;
                        if (ds) {
                            var field = ds.getField(dsField);
                            if (field && field.title) {
                                value = field.title;
                            }
                        }
                    }
                }
                return value;
            }
        },

        changed : function (form, item, value) {
            var grid = item.grid,
                rowNum = grid.getEditRow(),
                colNum = grid.getFieldNum("name"),
                nameValue = grid.getEditValue(rowNum, colNum),
                record = grid.getRecord(rowNum),
                overwriteName = (record && record._overwriteName)
            ;
            if (grid._updateFromTitle || overwriteName || !nameValue || nameValue == "") {
                var field = grid.getField("title");
                nameValue = isc.DataSourceEditor.createNameFromTitle(value);

                grid.setEditValue(rowNum, colNum, nameValue);
                grid._updateFromTitle = true;
                // _overwriteName flag only applies for starting the overwrite
                if (overwriteName) delete record._overwriteName;
            }
        },

        editorExit : function (editCompletionEvent, record, newValue, rowNum, colNum, grid) {
            delete grid._updateFromTitle;
        }
    },
    {name:"name", title: "Internal name", required: true, canHover: true,
        prompt: "Internal name for your field. " +
            "Users of your application will not see this name, but it will appear in " +
            "exported code and in some advanced areas of this design tool",
        // Where includeFrom has been used, the name defaults to includeFrom's name.
        // So as well show that instead of nothing. We'll put it in italics to indicate
        // that it is special.
        //
        // In fact, we may as well show the includeFrom value in all cases (where
        // present) -- this will help avoid confusion where the name has been edited.

        formatCellValue : function(value, record, rowNum, colNum, grid) {
            if (!record) record = {};
            var formattedValue = this._nameFromValueOrIncludeFrom(value, record.includeFrom);
            if (record.includeFrom) {
                var formattedName = grid.formatRelatedField(record.includeFrom),
                    formattedVia = (record.includeVia ? " via " + record.includeVia : ""),
                    formattedAggregation = (record.includeSummaryFunction ? " as " + record.includeSummaryFunction : "");
                formattedValue += " [Included from: <i>" + formattedName + formattedVia + formattedAggregation + "</i>]";
            } else if (record._inheritedFrom) {
                var formattedName = grid.createDSLink(record._inheritedFrom);
                formattedValue += " [Inherited from: <i>" + formattedName + "</i>]";
            }
            return formattedValue;
        },

        // If the value is present, return it. Otherwise, return the last
        // part of the includeFrom -- which is what the name defaults to.
        _nameFromValueOrIncludeFrom : function(value, includeFrom) {
            if (value || !includeFrom) {
                return value;
            } else {
                var dotIndex = includeFrom.lastIndexOf(".");
                if (dotIndex == -1) {
                    return value;
                } else {
                    return includeFrom.substring(dotIndex + 1);
                }
            }
        },

        // Note that name is *required* in the schema. This isn't literally true
        // anymore, since now name is optional when includesFrom is specified.
        // We could make it optional in the schema, but that may cause difficulties
        // elsewhere. So, for the moment, we're doing some massaging here.
        //
        // For editing, we'll display the last part of the includeFrom if the name
        // is blank -- that is what the default actually is, so it is a reasonable
        // starting point for editing.
        formatEditorValue : function(value, record, form, item) {
            if (!record) record = {};
            return this._nameFromValueOrIncludeFrom(value, record.includeFrom);
        },

        // If the user blanks the value, it would normally be an error (since
        // the name is required. So, let's simply supply the default in that
        // case -- that is, use the includeFrom's name. The alternative would
        // be to allow the blank, but that would mean changing the schema so that
        // name would not be required.
        parseEditorValue : function(value, record, rowNum, colNum, grid) {
            if (!record) record = {};
            return this._nameFromValueOrIncludeFrom(value, record.includeFrom);
        },

        showHover: true,
        hoverHTML : function (record, value, rowNum, colNum, grid) {
            var hover;
            if (record.includeFrom) {
                var split = record.includeFrom.split(".");
                if (split && split.length >= 2) {
                    var dsName = split[split.length-2],
                        fieldName = split[split.length-1]
                    ;
                    hover = "This field shows a value from the field <i>" + fieldName +
                        "</i> in the related DataSource <i>" + dsName + "</i>" +
                        "<p>To include a different field, just remove the included field " +
                        "and add a new one";
                }
            }
            return hover;
        },
        validators: [
            { 
                type: "regexp",
                expression: "^[a-zA-Z_][a-zA-Z0-9_]*$",
                errorMessage: "Field name must not contain spaces or punctuation other than underscore (_), and may not start with a number",
                stopIfFalse: true
            },
            {
                type:"custom",
                condition: function (item, validator, value, record, additionalContext) {
                    if (!value) return true;
                    var grid = additionalContext.component,
                        result = true;
                    ;
                    if (grid) {
                        value = value.toLowerCase();
                        var currentRow = additionalContext.rowNum,
                            totalRows = grid.getTotalRows()
                        ;
                        for (var i = 0; i < totalRows; i++) {
                            if (i == currentRow) continue;
                            var gridRecord = grid.getRecord(i),
                                name = (gridRecord && gridRecord[item.name] ? gridRecord[item.name].toLowerCase() : null)
                            ;
                            if (gridRecord && value == name) {
                                result = false;
                                break;
                            }
                        }
                    }
                    return result;
                },
                errorMessage: "Field name must be unique within the DataSource"
            }
        ]
    },
    {name:"type", width: 150, type: "ComboBoxItem",
        editorProperties: { addUnknownValues: false, completeOnTab: true },
        prompt: "Data type of this field",
        getEditorValueMap : function (values, field, grid) {
            return grid.creator.getFieldTypeValueMap();
        },
        formatCellValue : function(value, record, rowNum, colNum, grid) {
            if (!record) record = {};
            if (record.foreignKey) {
                var formattedName = grid.formatRelatedField(record.foreignKey);
                value = "relation (to " + formattedName + ")";
            } else if (record.includeFrom) {
                var session = grid.creator.creator.session;
                value = this._typeFromIncludeFrom(record.includeFrom, session);
            } else if (value == "sequence") {
                value = value + " (primary key)";
            }
            return value;
        },

        _typeFromIncludeFrom : function(includeFrom, session) {
            var type,
                split = includeFrom.split(".")
            ;
            if (split && split.length >= 2) {
                var dsName = split[split.length-2],
                    dsField = split[split.length-1],
                    ds = session.get(dsName)
                ;
                if (ds) {
                    var field = ds.getField(dsField);
                    type = field && field.type;
                }
            }
            return type;
        },

        change : function (form, item, value, oldValue) {
            var grid = this.grid || this,
                fieldEditor = grid.creator,
                dsEditor = fieldEditor.creator
            ;

            // Special handling for choosing a sequence field type
            if (value == "sequence") {
                // Only one sequence field is allowed
                var hasOtherSequenceFields = (grid.data.find("type", "sequence") != null);
                if (hasOtherSequenceFields) {
                    isc.say("DataSources may not have more than one sequence field. " +
                             "Please remove the other sequence field first or set it " +
                             "to be a normal 'integer' field.");
                    return false;
                }

                // Setting the field to a sequence will automatically move the primary key
                // as well. If there is a relation pointing to this DS confirm the user is
                // OK with removing the relation before continuing. This is only applicable
                // when changing the original PK field.
                //
                // The relation(s) won't be removed until this DS is saved. Additionally,
                // if the PK is moved back to the original field no relations are removed.
                //
                var rowNum = item.rowNum,
                    targetField = grid.getRecord(rowNum),
                    targetFieldName = targetField.name
                ;                

                // Existing records will assigned sequence numbers destroying
                // current values
                var confirmReplacingFieldValues = function (callback) {
                    var sampleData = (dsEditor.editSampleData ? dsEditor.getSampleData() : null) || [],
                        targetFieldValues = sampleData.getProperty(targetFieldName)
                    ;
                    var targetFieldHasValues = false;
                    for (var i = 0; i < targetFieldValues.length; i++) {
                        if (targetFieldValues[i] != null) {
                            targetFieldHasValues = true;
                            break;
                        }
                    }
                    if (targetFieldHasValues) {
                        // Field has data values that will be destroyed

                        isc.confirm("Setting this field to be a sequence will destroy any " +
                            "existing data in the field.  Continue?",
                            function (response)
                        {
                            if (response == true) {
                                callback();
                            }
                        }, {
                            buttons: [
                                isc.Dialog.CANCEL,
                                { title: "Destroy existing data", width:150, overflow: "visible",
                                    click: function () { this.topElement.okClick(); }
                                }
                            ],
                            autoFocusButton: 0
                        });

                        // callback will only be fired if user confirms above
                        return;
                    }
                    // field has no values to overwrite - no confirmation to provide
                    callback();
                };


                // Existing records will assigned sequence numbers destroying
                // current values
                var getChangePrimaryKeyAction = function (callback) {
                    var sampleData = (dsEditor.editSampleData ? dsEditor.getSampleData() : null) || [],
                        targetFieldValues = sampleData.getProperty(targetFieldName)
                    ;
                    var targetFieldHasValues = false;
                    for (var i = 0; i < targetFieldValues.length; i++) {
                        if (targetFieldValues[i] != null) {
                            targetFieldHasValues = true;
                            break;
                        }
                    }
                    if (targetFieldHasValues) {
                        // Determine if the new sequence field already has data in it that is a
                        // sequence. The schemaGuesser is used to check the data and determine
                        // the tightness.
                        var guesser = isc.SchemaGuesser.create({ detectPrimaryKey: true }),
                            guessedFields = guesser.extractFieldsFrom(sampleData) || [],
                            guessedTargetField = guessedFields.find("name", targetFieldName),
                            targetFieldDataIsSequence = (guessedTargetField && guessedTargetField.tightness == 0)
                        ;
                        if (targetFieldDataIsSequence) {
                            isc.confirm("Use the existing values in this field as the initial " +
                                        "sequence values or replace them with a new sequence " +
                                        "starting with 1?",
                                function (response)
                            {
                                if (response == true) {
                                    callback("keepAsSequence");
                                } else if (response == false) {
                                    callback("newSequence");
                                }
                            }, {
                                buttons: [
                                    isc.Dialog.CANCEL,
                                    { title: "Use existing values", width:150, overflow: "visible",
                                        click: function () { this.topElement.yesClick(); }
                                    },
                                    { title: "Create new values", width:150, overflow: "visible",
                                        click: function () { this.topElement.noClick(); }
                                    }
                                ],
                                autoFocusButton: 0
                            });

                            // callback will only be fired if user confirms above
                            return;
                        }
                    }
                    // field has no values to overwrite - no confirmation to provide
                    callback(null);
                };

                var applyPKChange = function (action) {
                    // Changing the primary key with action:"newSequence" will move the
                    // PK flag and change the field type to sequence. Additionally,
                    // field values will be updated with valid sequence values.
                    fieldEditor.changePrimaryKey(rowNum, targetField, action, function () {
                        // Update the button states using the current edit-state record
                        fieldEditor.updateButtonStates(grid.getEditedRecord(item.rowNum));
                    });
                };

                var pkFields = grid.data.findAll("primaryKey", true),
                    pkFieldIsSequence = (pkFields.find("type", "sequence") != null)
                ;
                // If another field is a sequence and it's a primaryKey, then confirm that
                // any relations can be removed
                if (pkFieldIsSequence) {
                    fieldEditor.confirmRemovingRelations(targetFieldName, function () {
                        getChangePrimaryKeyAction(function (action) {
                            if (action == null) {
                                confirmReplacingFieldValues(function () {
                                    applyPKChange("newSequence");
                                });
                            } else {
                                applyPKChange("newSequence");
                            }
                        });
                    });
                } else {
                    getChangePrimaryKeyAction(function (action) {
                        if (action == null) {
                            confirmReplacingFieldValues(function () {
                                applyPKChange("newSequence");
                            });
                        } else {
                            applyPKChange(action);
                        }
                    });
                }

                // Cancel update until asynchronous confirmations above are successful.
                // The change will then be explicitly applied.
                return false;
            }
        },

        changed : function (form, item, value) {
            // Update the button states using the current edit-state record
            item.grid.creator.updateButtonStates(item.grid.getEditedRecord(item.grid.getEditRow()));
        }
    },
    {name:"required", title:"Req.", width:40, canToggle:true, canHover: true,
        prompt: "Required: whether users are required to provide a value for this field when saving data",
        hoverHTML : function (record, value, rowNum, colNum, grid) {
            var hover;
            if (record.includeFrom) hover = "Included fields are not editable";
            return hover;
        }
    },
    {name:"hidden", width:60, canToggle:true,
        prompt: "Whether this field is hidden from users"
    },
    {name:"length", width:70,
        prompt: "For text fields, what is the maximum length allowed"
    },
    {name:"primaryKey", title:"Is PK", width:70, canToggle:true,
        prompt: "The Primary Key (PK) uniquely identifies each DataSource record and " +
                "allows related data across DataSources to be displayed together.",
        showHover: true,
        hoverHTML : function (record, value, rowNum, colNum, grid) {
            return this.prompt;
        },

        recordClick : function (viewer, record, recordNum, field, fieldNum, value, rawValue) {
            // Record recordNum for later use in change()
            
            viewer._lastRowNum = recordNum;
        },
        change : function (form, item, value, oldValue) {
            var grid = this.grid || this,
                fieldEditor = grid.creator,
                dsEditor = fieldEditor.creator,
                hasPkField = (grid.data.find("primaryKey", true) != null)
            ;

            // If no PK field is currently configured, let the normal assignment action
            // take place
            if (!hasPkField) return;

            // This shouldn't happen because canEditCell() disables editing the field
            // when not appropriate, however, just to make sure...
            if (!dsEditor.canChangePrimaryKey) return false;

            // Grab the row being unchecked
            
            var rowNum = (item ? item.rowNum : grid._lastRowNum),
                targetField = grid.getEditedRecord(rowNum),
                targetFieldName = targetField.name
            ;

            var checkChangedValue = function (rowNum, value) {
                if (value) {
                    // Setting primaryKey and maybe moving from another field
    
                    // Does this target field have valid data to be a PK? (i.e. unique and
                    // non-null values)
                    var sampleData = (dsEditor.editSampleData ? dsEditor.getSampleData() : null) || [],
                        targetFieldValues = sampleData.getProperty(targetFieldName),
                        targetFieldUniqueValues = targetFieldValues.getUniqueItems(),
                        valid = true
                    ;
                    if (targetFieldUniqueValues.length != targetFieldValues.length) {
                        // Non-unique values
                        valid = false;
                    }
    
                    if (valid) {
                        for (var i = 0; i < targetFieldValues.length; i++) {
                            if (targetFieldValues[i] == null) {
                                // null value
                                valid = false;
                            }
                        }
                    }
                    if (!valid) {
                        isc.confirm("To be used as a primary key, a field must have all " +
                                    "values unique and non-empty. Field <i>" +
                                    targetFieldName +
                                    "</i> does not - you can fix this in the Sample Data " +
                                    "tab",
                            function (response)
                        {
                            // OK means do nothing; cancel goes to sample data
                            if (response != true) {
                                dsEditor.mainTabSet.selectTab("sampleData");
                            }
                        }, {
                            buttons: [
                                { title: "Go to Sample Data", width:150, overflow: "visible",
                                    click: function () { this.topElement.cancelClick(); }
                                },
                                isc.Dialog.OK
                            ],
                            autoFocusButton: 0
                        });
    
                        // Cancel change to PK value
                        return false;
                    }
    
                    // Field is eligible for being a primary key
                    var options = {},
                        formValues = { action: "keepValues" }
                    ;
    
                    
    
                    if (sampleData.length > 0) {
                        if (targetField.type == "integer" || targetField.type == "sequence") {
                            
    
                            // Determine the next value in the "sequence"
                            var targetFieldValues = sampleData.getProperty(targetFieldName),
                                maxValue = Number.MIN_SAFE_INTEGER
                            ;
                            for (var i = 0; i < targetFieldValues.length; i++) {
                                var value = targetFieldValues[i];
                                if (value > maxValue) maxValue = value;
                            }
                            if (targetField.type == "sequence") {
                                options.keepAsSequence = "Keep existing sequence values for " +
                                    "<i>" + targetFieldName + "</i> with next value: " +
                                    (maxValue+1);
                            } else {
                                options.keepAsSequence = "Keep existing values and make <i>" +
                                    targetFieldName + "</i> a sequence with next value: " +
                                    (maxValue+1);
                            }
                            formValues.action = "keepAsSequence";
                        } else {
                            options.newSequence = "Discard old values and make <i>" +
                                targetFieldName + "</i> a " +
                                "sequence starting with 0";
                        }
                    } else {
                        if (targetField.type == "sequence") {
                            options.newSequence = "Keep <i>" + targetFieldName + "</i> " +
                                "a sequence starting with 0";
                        } else {
                            options.newSequence = "Make <i>" + targetFieldName + "</i> " +
                                "a sequence starting with 0";
                        }
                        formValues.action = "newSequence";
                    }
                    options.keepValues = (sampleData.length > 0 ?
                        "Keep values and r" :
                        "R") + "equire user to enter a new, unique value for <i>" + 
                        targetFieldName + "</i> when creating new records";

                    var dialog = isc.Dialog.create({
                        title: "Change primary key",
                        isModal: true,
                        autoSize: true,
                        autoCenter: true,
                        bodyDefaults: { padding: 10 },
                        saveData : function () {
                            var action = this.items[0].getValue("action");
                            fieldEditor.changePrimaryKey(rowNum, targetField, action);
                        }, 
                        items: [
                            isc.DynamicForm.create({
                                autoDraw: false,
                                autoFocus: true,
                                width: 600,
                                values: formValues,
                                items: [
                                    { name: "action", editorType: "RadioGroupItem",
                                        showTitle: false, width: "*", wrap: false,
                                        valueMap: options
                                    }
                                ]
                            }),
                            isc.LayoutSpacer.create({ autoDraw: false, height: 10 }),
                            isc.HLayout.create({
                                autoDraw: false,
                                height: 1,
                                membersMargin: 10,
                                width: 600,
                                align: "right",
                                members: [
                                    isc.IButton.create({
                                        autoShow: false,
                                        title: "Cancel",
                                        width: 75,
                                        click: function () { dialog.cancelClick(); }
                                    }),
                                    isc.IButton.create({
                                        autoShow: false,
                                        title: "OK",
                                        width: 75,
                                        click: function () { dialog.okClick(); }
                                    })
                                ]
                            })
                        ]
                    });
    
                    return false;
    
                } else {
                    // Removing primaryKey designation from a field.
    
                    var primaryKeyCount = grid.data.findAll("primaryKey", true).length;
    
                    // If there is only one primaryKey value there are conditions to removing
                    if (primaryKeyCount == 1) {
                        // If a primaryKey is required, see if there a new primaryKey field
                        // should be generated or if a previously generated one still exists
                        // to use
                        if (dsEditor.autoAddPK) {
                            // A PK should be automatically added or if a previous auto-added
                            // one can be found it can be used.
    
                            // Look for previous auto-added PK field. This will not handle
                            // the case where uniqueId is added but the renamed either by
                            // changing the DataSource ID or explicitly.
                            var field = grid.data.find("name", dsEditor.uniqueIdFieldName);
                            if (!field) field = grid.data.find("name", "internalId");
                            if (field && field.type == "sequence" && field.hidden == "true") {
                                var rowNum = grid.getRecordIndex(field);
                                grid.setEditValue(rowNum, "primaryKey", true);
                                return;
                            }
    
                            isc.confirm("All DataSources require a primary key. Do you want to add " +
                                        "an internal primary key field with automatically generated " +
                                        "values (the user will never see this field)?" +
                                        "<p>Note: to change to another existing field as primary " +
                                        "key, click the Primary Key checkbox for that field first.",
                                function (response)
                            {
                                if (response == true) {
                                    // Uncheck the primaryKey from the changing record
                                    grid.cancelEditing();
                                    var record = grid.getRecord(rowNum);
                                    record.primaryKey = false;
                                    grid.refreshRow(rowNum);
                                    grid.setEditValue(rowNum, "primaryKey", false);
    
                                    // Create a new PK field
                                    var fields = [];
                                    dsEditor.createUniqueIdField(fields);
                                    grid.creator.newRecord(fields[0]);

                                    // Save off the original primaryKey fieldName so we know whether
                                    // relations should be removed. They should if the PK has changed
                                    // and there are relations to this DS.
                                    if (!grid._originalPKFieldName) {
                                        grid._originalPKFieldName = record.name;
                                    }

                                    // If setting primary key back to the original field, we don't need
                                    // the original PK field name anymore - it hasn't changed.
                                    if (fields[0].name == grid._originalPKFieldName) {
                                        delete grid._originalPKFieldName;
                                    }
                                }
                            }, {
                                buttons: [
                                    isc.Dialog.CANCEL,
                                    { title: "Add Field", width:75, overflow: "visible",
                                      click: function () { this.topElement.okClick(); }
                                    }
                                ],
                                autoFocusButton: 1
                            });
                            // Don't allow the value change to take effect until the confirmation
                            // dialog has a positive response
                            return false;
    
                        } else if (dsEditor.requirePK) {
                            isc.say("All DataSources requires a Primary Key");
                            return false;
                        }
                    } else {
                        
                        return false;
                    }
                }
            };

            // If there is a relation pointing to this DS confirm the user is OK with
            // removing the relation before continuing. This is only applicable when
            // changing the original PK field.
            //
            // The relation(s) won't be removed until this DS is saved. Additionally,
            // if the PK is moved back to the original field no relations are removed.
            //
            fieldEditor.confirmRemovingRelations(targetFieldName, function () {
                checkChangedValue(rowNum, value);
            });

            // Cancel change to PK value
            return false;
        }
    }
],

getFieldEditorFormFields : function () {
    return isc.clone(this.fieldEditorFormFields);
},


fieldEditorFormFields : [
    {name:"name", canEdit:false},
    {name:"type"},
    {name:"multiple", 
        showIf:function() {
            var grid = this.form.creator,
                mainEditor = grid ? grid.creator.mainEditor : null,
                dsEditor = mainEditor && mainEditor.creator;
            return (dsEditor && dsEditor.canEditMultipleAttribute);
        }
    },
    {name:"title"},
    {name:"primaryKey"},
    {name:"valueXPath", colSpan:2, 
        showIf:function () {
            var grid = this.form.creator,
                mainEditor = grid ? grid.creator.mainEditor : null;
            return (mainEditor && mainEditor.getValues().dataFormat != 'iscServer');
            
        }
    },

    {type:"section", defaultValue:"Value Constraints",
     itemIds:["required", "length", "valueMap"] },
    {name:"valueMap", rowSpan:2},
    {name:"required"},
    {name:"length"},

    {type:"section", defaultValue:"Component Binding", 
     itemIds:["hidden", "detail", "canEdit"] },
    {name:"canEdit"},
    {name:"hidden"},
    {name:"detail"},

    {type:"section", defaultValue:"Relations", sectionExpanded:true,
     itemIds:["foreignKey", "rootValue", "includeFrom"] },
    {
        name: "foreignKey",
        type: "staticText",

        showPickerIcon: true,
        pickerConstructor: "DataSourceFieldPicker",
        pickerProperties: {
            width : 160,
            changed : function(form, value) {
                form.creator.setValue(value);
            }
        },

        // Override showPicker to set up the valid DataSources based on whatever
        // edits we've done.
        showPicker : function() { 
            // Look up the creator chain for the DataSourceEditor
            var dsEditor = this;
            while (dsEditor && !isc.isA.DataSourceEditor(dsEditor)) dsEditor = dsEditor.creator;
            if (!dsEditor) {
                this.logWarn("Could not find the DataSourceEditor");
                return;
            }
            var session = dsEditor.session,
                dataSources = session && session.getDataSources();
            if (!dataSources) {
                this.logWarn("DataSourceEditor.knownDataSources has not been set");
                return;
            }

            // Actually show the picker and set the valid Datasources
            this.Super("showPicker", arguments);    

            // Push session onto picker so chosen DataSources can be pulled
            // from the session in case live DataSources aren't being used
            this.picker.session = session;

            var dsData = dsEditor.getDatasourceData();
            
            var validDS = dataSources.getProperty("defaults").findAll({serverType: dsData.serverType});
            this.picker.setValidDsNames(validDS.getProperty("ID"));

            // Try to get the live DS we are editing, in case there are types defined on the DS
            var ds = session.get(dsData.ID);
            this.picker.requiredBaseType = isc.SimpleType.getBaseType(this.form.getValue("type"), ds);

            this.picker.setCombinedValue(this.getValue());
        },

        // Pickers aren't destroyed by default, so we'll do that here
        destroy : function() {
            if (this.picker) this.picker.destroy();
            this.Super("destroy", arguments);
        }
    },
    {name:"rootValue"},
    {
        name: "includeFrom", 
        type: "staticText", 

        showPickerIcon: true, 
        pickerConstructor: "DataSourceFieldPicker",
        pickerProperties: {
            changed : function(form, value) {
                form.creator.setValue(value);
            }
        },

        // Override showPicker to set up the valid DataSources based on whatever
        // edits we've done.
        showPicker : function() { 
            // Look up the creator chain for the DataSourceEditor
            var dsEditor = this;
            while (dsEditor && !isc.isA.DataSourceEditor(dsEditor)) dsEditor = dsEditor.creator;
            if (!dsEditor) {
                this.logWarn("Could not find the DataSourceEditor");
                return;
            }

            var dsData = dsEditor.getDatasourceData();

            // The known valid datasources are the ones for which we have a foreignKey
            // defined, since includeFrom only works one level at a time. So, as a first
            // approximation, we can just collect the foreignKey's we've defined for this
            // dataSource. Note that we get them from the form, rather than the real
            // dataSource, so that we can immediately react to any changes.
            var editedFieldData = dsData.fields;                

            var fieldsWithForeignKeys = editedFieldData.findAll(function (field) {
                return field.foreignKey;
            }) || [];
            var validDsNames = fieldsWithForeignKeys.map(function (field) {
                return field.foreignKey.split(".")[0];
            }).getUniqueItems();

            // It is also possible that foreignKeys can be guessed for other datasources,
            // but we won't know until they are chosen and lazily loaded. So, we also
            // supply a list of datasources of the same type as ours
            var allDsRecords = null;
            if (dsEditor.knownDataSources) {
                var ourType = dsData.serverType;
                if (ourType) {
                    allDsRecords = dsEditor.knownDataSources.findAll({dsType: ourType});
                } else {
                    allDsRecords = dsEditor.knownDataSources;
                }
            }

            // Actually show the picker (possibly creating it)
            this.Super("showPicker", arguments);

            this.picker.setValidDsNames(validDsNames);
            if (allDsRecords) this.picker.setAllDsRecords(allDsRecords);
                        
            // In order to allow validation of lazily loaded datasources, we provide
            // our ID and field records
            this.picker.setWarnIfNoForeignKey(dsData);
            
            this.picker.setCombinedValue(this.getValue());
        },

        // Pickers aren't destroyed by default, so we'll do that here
        destroy : function() {
            if (this.picker) this.picker.destroy();
            this.Super("destroy", arguments);
        }
    }
],

showOperationBindingsEditor:true,
opBindingsEditorDefaults: {
    _constructor: "DynamicForm",
    autoDraw: false,
    minHeight: 120,
    width: "100%",
    padding: 5,
    wrapItemTitles: false,

    initWidget : function () {
        this.fields = [
            { type: "blurb", defaultValue: "Edit the roles required to access each DataSource operation" },
            { type: "RowSpacer" },
            this.createRolesField("fetchRequiresRole", "Roles required for <i>fetch</i> operation"),
            this.createRolesField("addRequiresRole", "Roles required for <i>add</i> operation"),
            this.createRolesField("updateRequiresRole", "Roles required for <i>update</i> operation"),
            this.createRolesField("removeRequiresRole", "Roles required for <i>remove</i> operation")
        ];

        this.Super('initWidget', arguments);
    },

    specialFieldValues: ["_any_","*super*","false"],

    createRolesField : function (fieldName, title) {
        var availableRoles = isc.Auth.getAvailableRoles();
        var valueMap = {
            "_any_": "Any user - no roles required"
        };
        if (availableRoles) {
            availableRoles.sort();
            for (var i = 0; i < availableRoles.length; i++) {
                var role = availableRoles[i];
                valueMap[role] = role;
            }
        }
        valueMap["*super*"] = "SuperUser only";
        valueMap["false"] = "None - no user may access";

        var specialFieldValues = this.specialFieldValues;
        var field = {
            name: fieldName,
            type: "select",
            title: title,
            multiple: true,
            multipleAppearance: "picklist",
            valueMap: valueMap,
            pickListProperties: {
                selectionChanged : function (record, state) {
                    if (state) {
                        var value = record[fieldName];
                        if (specialFieldValues.contains(value)) {
                            // Selecting a special value, clear all other selections, if any
                            var records = this.getSelection();
                            for (var i = 0; i < records.length; i++) {
                                var record = records[i];
                                if (record[fieldName] != value) {
                                    this.deselectRecord(record);
                                }
                            }
                        } else {
                            // Selecting a role, clear any special value selections, if any
                            var records = this.getSelection();
                            for (var i = 0; i < records.length; i++) {
                                var record = records[i];
                                if (specialFieldValues.contains(record[fieldName])) {
                                    this.deselectRecord(record);
                                }
                            }
                        }
                    }
                }
            },
            editorEnter : function (form, item, value) {
                this._saveOldBinding(form, item, value);
            },
            setValue : function (newValue) {
                this._saveOldBinding(this.form, this, newValue);
                return this.Super("setValue", arguments);
            },
            _saveOldBinding : function (form, item, value) {
                var oldBinding;
                if (value) {
                    var dsData = form.creator.getDatasourceData(),
                        oldBindings = dsData.operationBindings,
                        operationType = item.name.replace("RequiresRole", ""),
                        matchingBindings = oldBindings && oldBindings.findAll("operationType", operationType)
                    ;
                    // Only use binding with no operationId
                    if (matchingBindings) {
                        for (var i = 0; i < matchingBindings.length; i++) {
                            if (!matchingBindings[i].operationId) {
                                oldBinding = matchingBindings[i];
                                break;
                            }
                        }
                    }
                }
                this._origBinding = oldBinding;
            },
            // editorExit : function (form, item, value) {
            changed : function (form, item, value) {
                var session = form.creator.session,
                    dsName = form.creator.getDataSourceID(),
                    operationType = item.name.replace("RequiresRole", "")
                ;

                var getBinding = function (binding, value) {
                    // combine original binding with the new value
                    binding = (binding ? isc.clone(binding) : {});
                    delete binding.requires;
                    delete binding.requiresRole;

                    if (isc.isAn.Array(value) && value.length == 1) value = value[0];

                    // Process special values
                    if (value == "false") {
                        binding.requires = "false";
                    } else if (value && value != "_any_") {
                        // Convert array of selections to a comma-separated string
                        if (isc.isAn.Array(value)) {
                            value = value.join(",");
                        }
                        binding.requiresRole = value;
                    }
                    return (isc.isAn.emptyObject(binding) ? null : binding);
                };

                var origBinding = this._origBinding,
                    newBinding = getBinding(this._origBinding, value)
                ;

                if (!origBinding && newBinding != null) {
                    // added
                    session.addChange(dsName, "addOperationBinding", {
                        operationType: operationType,
                        binding: newBinding
                    });
                } else if (origBinding != null && !newBinding) {
                    // removed
                    session.addChange(dsName, "removeOperationBinding", {
                        operationType: operationType,
                        binding: this._origBinding
                    });
                } else if (!isc.Canvas.compareValues(origBinding, newBinding)) {
                    // changed
                    var context;
                    if ((!origBinding.requires && newBinding.requires) ||
                        (origBinding.requires && !newBinding.requires))
                    {
                        context = session.addChange(dsName, "changeOperationBindingProperty", {
                            operationType: operationType,
                            property: "requires",
                            originalValue: origBinding.requires,
                            newValue: newBinding.requires
                        });
                    }

                    if ((!origBinding.requiresRole && newBinding.requiresRole) ||
                        (origBinding.requiresRole && !newBinding.requiresRole) ||
                        (origBinding.requiresRole && !origBinding.requiresRole.equals(newBinding.requiresRole)))
                    {
                        if (context) session.startChangeContext(context);
                        session.addChange(dsName, "changeOperationBindingProperty", {
                            operationType: operationType,
                            property: "requiresRole",
                            originalValue: origBinding.requiresRole,
                            newValue: newBinding.requiresRole
                        });
                        if (context) session.endChangeContext(context);
                    }
                }
                this._origBinding = newBinding;
            }
        }
        return field;
    }

},

mockEditorDefaults: {
    _constructor: "DynamicForm",
    minWidth: 80,
    minHeight: 20,
    width: "100%",
    height: "100%",
    numCols: 3,
    colWidths: [10, 150, "*"],
    fields: [
        {
            name:"ID", title: "Name", wrapTitle: false, colSpan: 2, required:true,
            selectOnFocus: true,
            hoverWidth: 300,
            validateOnExit: true,
            validators: [
                { 
                    type: "regexp",
                    expression: "^(?!isc_).*$",
                    errorMessage: "DataSource ID must not start with 'isc_'. That prefix is reserved for framework DataSources.'",
                    stopIfFalse: true
                },
                { 
                    type: "regexp",
                    expression: "^[a-zA-Z_][a-zA-Z0-9_]*$",
                    errorMessage: "DataSource ID must not contain spaces or punctuation other than underscore (_), and may not start with a number",
                    stopIfFalse: true
                },
                {
                    type:"custom",
                    condition: function (item, validator, value, record, additionalContext) {
                        if (!value) return true;
                        if (!validator.idMap) {
                            // Create idMap to map from lowercase ID to actual ID so that
                            // entered name can be matched to an existing schema regardless
                            // of case.
                            var allDataSources = isc.DS.getRegisteredDataSourceObjects(),
                                idMap = {}
                            ;
                            for (var i = 0; i < allDataSources.length; i++) {
                                var ds = allDataSources[i];
                                if (ds && ds.componentSchema) {
                                    var id = ds.ID;
                                    idMap[id.toLowerCase()] = id;
                                }
                            }
                            validator.idMap = idMap;
                        }
                        // Also validate that the non-schema DS is not a SimpleType and not
                        // an internal application DS (i.e. a Reify-created DS is OK)
                        var id = validator.idMap[value.toLowerCase()] || value,
                            ds = item.form.creator.session.get(id),
                            isProjectDataSource = ds && (ds.sourceDataSourceID != null ||
                                item.form.creator.session.getDataSourceDefaults(id) != null)
                        ;
                        return ((!ds || (!ds.componentSchema && isProjectDataSource)) &&
                                !isc.SimpleType.getType(value));
                    },
                    errorMessage: "DataSource name matches a system or application DataSource. Please choose another name."
                }
            ]
        },
        {
            name: "edit",
            type: "TextArea",
            colSpan: 3,
            allowNativeResize: true,
            width: "*", height: "*",
            showTitle: false,
            browserSpellCheck: false,
            changed : function (form, item, value) {
                form.editDataChanged(value);
            }
        },
        {
            name: "addField", showTitle: false, title: "add a unique ID field automatically", type: "checkbox", colSpan: 3,
            changed : function (form, item, value) {
                if ( value ) form.setValue( "useField", null );
                form.validate();
            }
        },
        {
            name: "useField", showTitle: false, title: "use", type: "checkbox", width: 10,
            changed : function (form, item, value) {
                if ( value ) form.setValue( "addField", null );
            }
        },
        {
            name: "fieldName", showTitle: false, hint: "field", showHintInField: true,
            width: 150,
            editorType: "ComboBoxItem",
            addUnknownValues: false,
            valueField: "name",
            getClientPickListData : function () {
                var fileType = this.form.getValue("mockDataFormat"),
                    rawData = this.form.getValue("edit"),
                    fields = this.form.getParsedFields(fileType, rawData)
                ;
                var data = [];
                if (!fields || fields.length == 0) {
                    this.setErrors("No fields could be detected in the data");
                    this.setValue(null);
                    return data;
                }
                this.clearErrors();
                for (var i = 0; i < fields.length; i++) {
                    data.add({ name: fields[i].name });
                }
                return data;
            },
            changed : function (form, item, value) {
                form.setValue( "addField", null );
                form.setValue( "useField", true );
            },
            requiredWhen: {
                fieldName: "useField", operator: "equals", value: true
            }
        },
        { name: "useField2", type: "StaticTextItem", showTitle: false, defaultValue: "as the unique ID field" }
    ],
    initWidget : function () {
        if (!this.allowExplicitPKInSampleData) {
            this.fields = isc.clone(this.fields);
            this.fields.find("name", "addField").visible = false;
            this.fields.find("name", "useField").visible = false;
            this.fields.find("name", "fieldName").visible = false;
            this.fields.find("name", "useField2").visible = false;
        }
        this.Super("initWidget", arguments);
    },
    editDataChanged : function (value) {
        if (!this.allowExplicitPKInSampleData) return;
        var haveData = false;
    
        var fileType = this.getValue("mockDataFormat");
        if (fileType == "json" || fileType == "csv" || fileType == "xml") {
            value = (value ? value.trim() : null);
            value = (value != "" ? value : null);
            haveData = (value != null);
        }
        if (haveData) {
            this.getField("addField").enable();
            this.getField("useField").enable();
            this.getField("fieldName").enable();
        } else {
            this.getField("addField").disable();
            this.getField("useField").disable();
            this.getField("fieldName").disable();
        }
    },
    getParsedFields : function (fileType, rawData) {
        var parse = function (data) {
            var parsedData;
            // Parse data
            var parser = isc.FileParser.create({ hasHeaderLine: true });
            if (fileType == "json" || fileType == "xml") {
                // XML data is pre-processed into JSON before getting here
                parsedData = parser.parseJsonData(data); 
                if (parsedData._parseFailure) {
                    return null;
                }
                if (parsedData._notAnArray) {
                    return null;
                }
            } else if (fileType == "csv") {
                parsedData = parser.parseCsvData(data);
            }
            return parser.getFields();
        };
    
        if (fileType == "xml") {
            // Process XML data into JSON.
            var xmlData = isc.xml.parseXML(rawData);
            var elements = isc.xml.selectNodes(xmlData, "/"),
                jsElements = isc.xml.toJS(elements)
            ;
            if (jsElements.length == 1) {
                var encoder = isc.JSONEncoder.create({ dateFormat: "dateConstructor", prettyPrint: false });
                var json = encoder.encode(jsElements[0]);
                var err = this.xmlParserError(isc.JSON.decode(json));
                if (err) {
                    return null;
                }
                return parse(json);
            }
        } else {
            return parse(rawData);
        }
        return null;
    },
    xmlParserError : function(obj, parserErrorSeen) {
        // If the passed-in object is a JS representation of markup showing an XML parser error,
        // returns the error text (otherwise null)
        if (!isc.isAn.Object(obj) || isc.isAn.Array(obj)) return;
        for (var attr in obj) {
            if (attr == "parsererror") {
                parserErrorSeen = true;
            }
            if (attr == "xmlTextContent" && parserErrorSeen) {
                return obj[attr];
            }
            var subErr = this.xmlParserError(obj[attr], parserErrorSeen);
            if (subErr) return subErr;
        }
    }
},

newButtonDefaults:{
    _constructor:"IButton",
    autoParent:"gridButtons",
    title: "New Field",
    width: 150,
    click:"this.creator.newRecord()"
},

//> @attr dataSourceEditor.moreButton (AutoChild Button : null : IR)
// "More" button autoChild to show additional editing properties for the selected
// field.
// @visibility reifyOnSite
//<
moreButtonDefaults:{
    _constructor:"IButton",
    autoParent:"gridButtons",
    width: 150,
    click:"this.creator.editMore()",
    disabled:true
},

buttonLayoutDefaults: {
    _constructor: "HLayout",
    height:42,
    layoutMargin:10,
    membersMargin:10,
    align: "right"
},

cancelButtonDefaults: {
    _constructor: "IButton",
    autoDraw: false,
    title: "Cancel",
    width: 100,
    autoParent: "buttonLayout",
    click: function() {
        this.creator.cancelClick();
    }
},

saveButtonDefaults: {
    _constructor: "IButton",
    autoDraw: false,
    title: "Save",
    width: 100,
    autoParent: "buttonLayout",
    click: function() {
        this.creator.saveClick();
    }
},

addTestDataButtonDefaults: {
    _constructor: "IButton",
    autoDraw: false,
    title: "Add Test Data",
    autoFit: true,
    click: function(){
        var dsData = isc.addProperties({}, 
            this.creator.mainEditor ? this.creator.mainEditor.getValues() : this.creator.mainEditorValues
        )
        var dataImportDialog = isc.DataImportDialog.create({
            ID: "dataImportDialog",
            targetDataSource: dsData.ID
        });
        dataImportDialog.show();
    }
},

editWithFieldsButtonDefaults: {
    _constructor: "IButton",
    autoDraw: false,
    title: "Edit fields and data separately...",
    autoFit: true,
    click: function() {
        this.creator.switchToEditFieldsAndDataSeparately();
    }
},

legalValuesButtonDefaults: {
    _constructor: "IButton",
    autoDraw: false,
    title: "Allowed values list",
    width: 150,
    disabled: true,
    click: function() {
        var editor = this.creator.fieldEditor,
            grid = editor.grid,
            tree = grid.data,
            selectedNode = grid.getSelectedRecord() || tree.root,
            parentNode = (isc.isA.Tree(tree) ? tree.getParent(selectedNode) : null)
        ;

        if (selectedNode && !selectedNode.isFolder && parentNode == tree.root) {
            // Look up the creator chain for the DataSourceEditor
            var dsEditor = this;
            while (dsEditor && !isc.isA.DataSourceEditor(dsEditor)) dsEditor = dsEditor.creator;
            if (!dsEditor) {
                this.logWarn("Could not find the DataSourceEditor");
                return;
            }
            dsEditor.editFieldLegalValues(selectedNode);
        }
    },
    canHover: true,
    hoverStyle: "vbLargeHover",
    getHoverHTML : function () {
        return (this.isDisabled() ? "To create a field that only allows specific values, select field type 'enum'" : null);
    }
},

derivedValueButtonDefaults: {
    _constructor: "IButton",
    autoDraw: false,
    title: "Derived value...",
    // icon: "[SKINIMG]actions/accept.png",
    iconAlign: "right",
    iconSize: 10,
    width: 150,
    disabled: true,
    click: function() {
        var editor = this.creator.fieldEditor,
            grid = editor.grid,
            tree = grid.data,
            selectedNode = grid.getSelectedRecord() || tree.root,
            parentNode = (isc.isA.Tree(tree) ? tree.getParent(selectedNode) : null)
        ;

        if (selectedNode && !selectedNode.isFolder && parentNode == tree.root) {
            // Look up the creator chain for the DataSourceEditor
            var dsEditor = this;
            while (dsEditor && !isc.isA.DataSourceEditor(dsEditor)) dsEditor = dsEditor.creator;
            if (!dsEditor) {
                this.logWarn("Could not find the DataSourceEditor");
                return;
            }
            dsEditor.editDerivedValue(selectedNode);
        }
    }
},

validatorsButtonDefaults: {
    _constructor: "IButton",
    autoDraw: false,
    title: "Validators...",
    width: 150,
    disabled: true,
    click: function() {
        var editor = this.creator.fieldEditor,
            grid = editor.grid,
            tree = grid.data,
            selectedNode = grid.getSelectedRecord() || tree.root,
            parentNode = (isc.isA.Tree(tree) ? tree.getParent(selectedNode) : null)
        ;

        if (selectedNode && !selectedNode.isFolder && parentNode == tree.root) {
            // Look up the creator chain for the DataSourceEditor
            var dsEditor = this;
            while (dsEditor && !isc.isA.DataSourceEditor(dsEditor)) dsEditor = dsEditor.creator;
            if (!dsEditor) {
                this.logWarn("Could not find the DataSourceEditor");
                return;
            }
            dsEditor.editFieldValidators(selectedNode);
        }
    }
},

securityButtonDefaults: {
    _constructor: "IButton",
    autoDraw: false,
    title: "Security...",
    width: 150,
    disabled: true,
    click: function() {
        var editor = this.creator.fieldEditor,
            grid = editor.grid,
            tree = grid.data,
            selectedNode = grid.getSelectedRecord() || tree.root,
            parentNode = (isc.isA.Tree(tree) ? tree.getParent(selectedNode) : null)
        ;

        if (selectedNode && !selectedNode.isFolder && parentNode == tree.root) {
            // Look up the creator chain for the DataSourceEditor
            var dsEditor = this;
            while (dsEditor && !isc.isA.DataSourceEditor(dsEditor)) dsEditor = dsEditor.creator;
            if (!dsEditor) {
                this.logWarn("Could not find the DataSourceEditor");
                return;
            }
            dsEditor.editFieldSecurity(selectedNode);
        }
    }
},

addChildButtonDefaults: {
    _constructor: "IButton",
    autoDraw: false,
    title: "Add Child Object",
    width: 150,
    click: function() {
        var editor = this.creator.fieldEditor,
            grid = editor.grid,
            tree = grid.data,
            selectedNode = grid.getSelectedRecord() || tree.root,
            parentNode = tree.getParent(selectedNode),
            newNode = {
                isFolder: true,
                children: [],
                multiple: true,
                childTagName: "item"
            }
        ;

        if (selectedNode) {
            if (!selectedNode.isFolder) selectedNode = parentNode;
            newNode.name = editor.getNextUniqueFieldName(selectedNode, "child"),
            newNode.id = editor.getNextUnusedNodeId(),
            newNode.parentId = selectedNode.id;
            // Let title field know that name can be overwritten
            newNode._overwriteName = true;
            tree.linkNodes([newNode], parentNode);
            tree.openFolder(newNode);
            // Auto-edit new child node
            var node = grid.findByKey(newNode.id);
            if (node) {
                var rowNum = grid.getRecordIndex(node);
                if (rowNum >= 0) {
                    grid.startEditing(rowNum);
                }
            }
        }

    }
},

relationsButtonDefaults: {
    _constructor: "IButton",
    autoDraw: false,
    title: "Relations...",
    prompt: "Relations are links between records of different DataSources such as a link " +
            "from an Order to the Customer that ordered it.<P>" +
            "Click the <i>Relations</i> button to view & edit relations.",
    hoverStyle: "vbLargeHover",
    width: 150,
    disabled: true,
    click: function() {
        // Look up the creator chain for the DataSourceEditor
        var dsEditor = this;
        while (dsEditor && !isc.isA.DataSourceEditor(dsEditor)) dsEditor = dsEditor.creator;
        if (!dsEditor) {
            this.logWarn("Could not find the DataSourceEditor");
            return;
        }
        dsEditor.editRelations();
    }
},

includeFieldButtonDefaults: {
    _constructor: "IButton",
    autoDraw: false,
    title: "Include field...",
    width: 150,
    prompt: "Add a field whose data comes from fields in a related DataSource",
    hoverWidth: 125,
    hoverStyle: "vbLargeHover",
    disabled: true,
    click: function() {
        // Look up the creator chain for the DataSourceEditor
        var dsEditor = this;
        while (dsEditor && !isc.isA.DataSourceEditor(dsEditor)) dsEditor = dsEditor.creator;
        if (!dsEditor) {
            this.logWarn("Could not find the DataSourceEditor");
            return;
        }
        dsEditor.editIncludeField();
    }
},

formattingButtonDefaults: {
    _constructor: "IButton",
    autoDraw: false,
    title: "Formatting...",
    disabled: true,
    width: 150,
    click: function() {
        // Look up the creator chain for the DataSourceEditor
        var editor = this.creator.fieldEditor,
            grid = editor.grid,
            tree = grid.data,
            selectedNode = grid.getSelectedRecord() || tree.root,
            parentNode = (isc.isA.Tree(tree) ? tree.getParent(selectedNode) : null)
        ;

        if (selectedNode && !selectedNode.isFolder && parentNode == tree.root) {
            var dsEditor = this;
            while (dsEditor && !isc.isA.DataSourceEditor(dsEditor)) dsEditor = dsEditor.creator;
            if (!dsEditor) {
                this.logWarn("Could not find the DataSourceEditor");
                return;
            }
            dsEditor.editFormatting(selectedNode);
        }
    }
},

reorderButtonLayoutDefaults: {
    _constructor: "HLayout",
    membersMargin: 4,
    layoutAlign: "center",
    height: 10
},

reorderUpButtonDefaults: {
    _constructor: "ImgButton",
    src: "[SKINIMG]TransferIcons/up.png",
    width: 22, height: 22,
    prompt: "Move field up",
    click: function() {
        // Look up the creator chain for the DataSourceEditor
        var editor = this.creator.fieldEditor,
            grid = editor.grid,
            tree = grid.data,
            selectedNode = grid.getSelectedRecord() || tree.root,
            parentNode = (isc.isA.Tree(tree) ? tree.getParent(selectedNode) : null)
        ;

        if (selectedNode && !selectedNode.isFolder && parentNode == tree.root) {
            var dsEditor = this;
            while (dsEditor && !isc.isA.DataSourceEditor(dsEditor)) dsEditor = dsEditor.creator;
            if (!dsEditor) {
                this.logWarn("Could not find the DataSourceEditor");
                return;
            }
            dsEditor.moveField(selectedNode, -1);
        }
    }
},

reorderDownButtonDefaults: {
    _constructor: "ImgButton",
    src: "[SKINIMG]TransferIcons/down.png",
    width: 22, height: 22,
    prompt: "Move field down",
    click: function() {
        // Look up the creator chain for the DataSourceEditor
        var editor = this.creator.fieldEditor,
            grid = editor.grid,
            tree = grid.data,
            selectedNode = grid.getSelectedRecord() || tree.root,
            parentNode = (isc.isA.Tree(tree) ? tree.getParent(selectedNode) : null)
        ;

        if (selectedNode && !selectedNode.isFolder && parentNode == tree.root) {
            var dsEditor = this;
            while (dsEditor && !isc.isA.DataSourceEditor(dsEditor)) dsEditor = dsEditor.creator;
            if (!dsEditor) {
                this.logWarn("Could not find the DataSourceEditor");
                return;
            }
            dsEditor.moveField(selectedNode, 1);
        }
    }
},

mainTabSetDefaults: {
    _constructor: "TabSet",
    overflow: "visible",
    width: "100%", height:"100%"
},

mainStackDefaults: {
    _constructor: "SectionStack",
    overflow: "visible",
    width: "100%", height:"100%",
    visibilityMode: "multiple"
},

instructionsSectionDefaults: {
    _constructor: "SectionStackSection",
    title: "Instructions",
    expanded:true, canCollapse:true,
    hidden:true // hidden by default
},

instructionsDefaults: {
    _constructor: "HTMLFlow", 
    autoFit:true,
    padding:10
},

mainSectionDefaults: {
    _constructor: "SectionStackSection",
    title:"DataSource Properties", 
    expanded:true, canCollapse:false, showHeader: false
},

fieldSectionDefaults: {
    _constructor: "SectionStackSection",
    title:"DataSource Fields &nbsp;<span style='color:#BBBBBB'>(click to edit or press New Field)</span>", 
    expanded:true, canCollapse: true
},

opBindingsSectionDefaults: {
    _constructor: "SectionStackSection",
    name:"operationBindings",
    title:"DataSource Operations", 
    expanded:false
},

mockSectionDefaults: {
    _constructor: "SectionStackSection",
    title:"MockDataSource Data", 
    expanded:true, canCollapse:false, showHeader: false, hidden: true
},

deriveFieldsSectionDefaults: {
    _constructor: "SectionStackSection",
    title:"Derive Fields From SQL",
    expanded:false, canCollapse: true
},

// Sample Data tab
sampleDataPaneDefaults: {
    _constructor: isc.VLayout,
    autoDraw: false,
    width: "100%",
    height: "100%"
},

sampleDataLabelDefaults: {
    _constructor: isc.Label,
    autoDraw: false,
    width: "100%",
    height: 35,
    padding: 10,
    contents: "Type in sample data below"
},

sampleDataGridDefaults: {
    _constructor: isc.ListGrid,
    autoDraw: false,
    width: "100%",
    height: "100%",
    // A filter is not applicable because all data is pre-saved so it gets validated
    // showFilterEditor: true,
    canEdit: true,
    canRemoveRecords: true,
    editEvent: "click",
    listEndEditAction: "next",
    escapeKeyEditAction: "done",
    autoSaveEdits: false,
    warnOnUnmappedValueFieldChange: false,
    startEditing: function(rowNum, colNum, suppressFocus) {
        var record = this.getEditedRecord(rowNum);
        if (isc.getKeys(record) < 1) {
            this.setEditValue(rowNum, '_operation', 'add');
        } else if (!record._operation) {
            this.setEditValue(rowNum, '_operation', 'update');
        }
        this.Super('startEditing', arguments);
    },
    
    validateRow : function (rowNum, suppressRefresh) {
        var result = this.Super("validateRow", arguments),
            pkFieldValue = (this._pkFieldName ? this.getEditValue(rowNum, this._pkFieldName) : null)
        ;
        if (this._pkFieldName && pkFieldValue == null) {
            this.setEditValue(rowNum, this._pkFieldName, this._pkNextValue++);
        }

        return result;
    }
},

sampleDataButtonLayoutDefaults: {
    _constructor: isc.HLayout,
    autoDraw: false,
    width: "100%", height: 35,
    padding: 5,
    membersMargin: 10
},

sampleDataAddRecordButtonDefaults: {
    _constructor: isc.Button,
    autoDraw: false,
    title: "Add New Record",
    autoFit: true,
    click : function () {
        this.creator.dataGrid.startEditingNew();
    }
},

sampleDataDiscardDataButtonDefaults: {
    _constructor: isc.Button,
    autoDraw: false,
    wrap: false,
    autoFit: true,
    title: "Discard sample data changes",
    click : function () {
        this.creator.revertDataChanges();
    }
},

// Window "body"
bodyProperties:{
    overflow:"auto",
    layoutMargin:10
},

deriveFormDefaults: {
    _constructor: "DynamicForm"
},

previewGridDefaults: {
    _constructor: "ListGrid",
    showFilterEditor: true    
},

// ---------------------------------------------------------------------------------------
// properties

//> @attr dataSourceEditor.canEditDataSourceID (Boolean : true : IRW)
// Can the DataSource ID be edited?
//
// @visibility reifyOnSite
//<
canEditDataSourceID: true,

//> @attr dataSourceEditor.showInheritedFields (Boolean : null : IRW)
// Should fields inherited from a parent DataSource be shown?
//
// @visibility reifyOnSite
//<

//> @attr dataSourceEditor.showWarningOnRename (Boolean : null : IRW)
// Should a +link{renameWarningMessage,warning} be shown when a DataSource ID is changed?
//
// @visibility reifyOnSite
//<

//> @attr dataSourceEditor.renameWarningMessage (Boolean : "Renaming a DataSource can <b>break</b> relations.<P><b>This operation cannot be undone.</b><P>Rename anyway?" : IRW)
// Message that should be shown when +link{showWarningOnRename,warning} a user of the
// consequences of a DataSource rename.
//
// @group i18nMessages
// @visibility reifyOnSite
//<
renameWarningMessage: "Renaming a DataSource can <b>break</b> relations.<P>" +
                      "<b>This operation cannot be undone.</b><P>" +
                      "Rename anyway?",

//> @attr dataSourceEditor.canAddChildSchema (Boolean : false : IRW)
// Can a child schema be added to a field?
//
// @visibility reifyOnSite
//<
canAddChildSchema: false,

//> @attr dataSourceEditor.canEditChildSchema (Boolean : false : IRW)
// Can a child schema be edited on a field?
//
// @visibility reifyOnSite
//<
canEditChildSchema: false,

//> @attr dataSourceEditor.canSelectPrimaryKey (Boolean : true : IRW)
// Can a field be selected as a primary key? Only a single field can be selected as
// primary key.
//
// @visibility reifyOnSite
//<
canSelectPrimaryKey: true,

//> @attr dataSourceEditor.canNavigateToDataSource (Boolean : true : IRW)
// When a relationship is shown for a field, should a link be shown to navigate
// to the referenced DataSource?
// <P>
// If a relationship link is clicked, +link{navigateToDataSource} is called.
//
// @visibility reifyOnSite
//<
canNavigateToDataSource: true,

//> @attr dataSourceEditor.showActionButtons (Boolean : true : IRW)
// Show +link{cancelButton,"Cancel"} and +link{saveButton,"Save"} action buttons
// for the editor as a whole?
// <p>
// If action buttons are not shown it is the responsibility of the caller to
// call +link{save} to persist changes if desired. See +link{editComplete} for
// a description of the save process.
//
// @visibility reifyOnSite
//<
showActionButtons: true,

//> @attr dataSourceEditor.showMoreButton (Boolean : true : IRW)
// Show +link{moreButton,"More" button} for editing field details?
//
// @visibility reifyOnSite
//<

//> @attr dataSourceEditor.showLegalValuesButton (Boolean : null : IRW)
// Show "Allowed values list" button for editing field enum values?
//
// @visibility reifyOnSite
//<

//> @attr dataSourceEditor.showDerivedValueButton (Boolean : true : IRW)
// Show "Derived value" button for editing field value?
//
// @visibility reifyOnSite
//<
showDerivedValueButton: true,

//> @attr dataSourceEditor.showIncludeFromButton (Boolean : null : IR)
// Should includeFrom field editor be available while editing the DataSource?
//
// @visibility reifyOnSite
//<

//> @attr dataSourceEditor.requirePK (Boolean : null : IRW)
// Is a primary key field required (locally or on a parent) for DataSources?
//
// @visibility reifyOnSite
//<

//> @attr dataSourceEditor.canChangePrimaryKey (Boolean : true : IRW)
// Can a primary key value be changed? This is different from +link{canSelectPrimaryKey} which
// controls whether the primary key property is shown or not.
//
// @visibility reifyOnSite
//<
canChangePrimaryKey: true,

//> @attr dataSourceEditor.autoAddPK (Boolean : null : IRW)
// Create a primary key field if there is no existing one defined. Additionally,
// this automatic field cannot be removed as a primary key nor can it be removed
// completely. An automatic primary key field is always given the +link{uniqueIdFieldName} initially but
// can be renamed. 
//
// @visibility reifyOnSite
//<

//> @attr dataSourceEditor.makeUniqueTableName (Boolean : null : IRW)
// When saving the edited DataSource should the +link{dataSource.tableName} be checked
// against the DB to confirm it is unique. If not unique a suffix
// will be added to guarantee uniqueness.
//
// @visibility reifyOnSite
//<

//> @attr dataSourceEditor.editMockData (Boolean : null : IRW)
// When editing a MockDataSource only a text field is presented to enter
// the +link{MockDataSource.mockData} text unless explicit fields are
// defined. To force editing of fields instead of <code>mockData</code>
// set this property to <code>false</code>.
//
// @visibility reifyOnSite
//<

//> @attr dataSourceEditor.editSampleData (Boolean : null : IRW)
// Should a sample data tab be added to allow editing of +link{dataSource.cacheData,cacheData}
// for DataSource?
//
// @visibility reifyOnSite
//<

// ---------------------------------------------------------------------------------------
// methods

//> @method dataSourceEditor.editNew
// Starts editing a new DataSource where <code>dataSource</code> specifies the starting
// properties of the new DataSource.
// <p>
// If +link{showActionButtons} are enabled, the Save and Cancel buttons will call
// +link{editComplete} by default (see +link{saveClick} and +link{cancelClick}).
// The caller can then determine what to do with the changes, if any.
// <P>
// See +link{editComplete} for details.
//
// @param [dataSource] (DataSource | PaletteNode) the dataSource to be edited
// @visibility reifyOnSite
//<
editNew : function (dataSource) {
    this.addTestDataButton.hide();
    this.editWithFieldsButton.hide();

    if (this.knownDataSources && this.knownDataSources.length > 0 && !this.session.dataSources) {        
        this.session.setDataSources(this.knownDataSources);
    }

    if (this.editSampleData) {
        this._rebindSampleDataGrid();
        this.originalSampleData = null;
    }

    if (!dataSource || dataSource.defaults) {
        var defaults = (dataSource ? dataSource.defaults : { _constructor: "DataSource "});
        this.paletteNode = dataSource;
        // Yes, pass defaults as dataSource and defaults
        this.start(defaults, defaults, true);
    } else {
        this.start(dataSource, null, true);
    }
},
    
//> @method dataSourceEditor.editSaved
// Starts editing a previously saved (existing) DataSource.
// <p>
// If +link{showActionButtons} are enabled, the Save and Cancel buttons will call
// +link{editComplete} by default (see +link{saveClick} and +link{cancelClick}).
// <P>
// The caller can then determine what to do with the changes, if any.
// See +link{editComplete} for details.
//
// @param dataSource (DataSource | ID | Object) the dataSource or defaults to be edited
// @visibility reifyOnSite
//<
editSaved : function (dataSource) {
    if (!dataSource) return;

    var defaults;
    if (this.knownDataSources && !this.session.dataSources) {
        this.session.setDataSources(this.knownDataSources);
    }

    if (isc.isA.String(dataSource)) {
        var self = this;
        this.session.loadDataSourceDefaults(dataSource, function (defaults) {
            var liveDS = self.session.get(dataSource);
            // If knownDataSources didn't include this DS and we're not using live dataSources
            // create a paletteNode for it dynamically.
            
            if (liveDS == null && !this.useLiveDataSources && !this.knownDataSources) {
                self.session._addToDataSources({
                    ID:dataSource,
                    defaults:defaults
                })
                liveDS = self.session.get(dataSource);
            }
            self._editSaved(liveDS, defaults);
        });
        return;
    } else if (!isc.isA.DataSource(dataSource)) {
        // Create a temporary live DS instance from the defaults for editing.
        // Defaults can be provided via editNode/paletteNode or a just a defaults object
        defaults = dataSource.defaults || dataSource;
        var dsName = defaults && defaults.ID;
        // Last parameter indicates that this is not a "new" dataSource
        // requiring an add operation to save changes.
        this.session.add(dsName, defaults, true);
        dataSource = this.session.get(dsName);
    }

    this._editSaved(dataSource, defaults);
},

//> @method dataSourceEditor.editSavedFromXml
// Starts editing a previously saved (existing) DataSource using Xml source.
// <p>
// If +link{showActionButtons} are enabled, the Save and Cancel buttons will call
// +link{editComplete} by default (see +link{saveClick} and +link{cancelClick}).
// <P>
// The caller can then determine what to do with the changes, if any.
// See +link{editComplete} for details.
//
// @param dataSource (DataSource | ID | Object) the dataSource or defaults to be edited
// @visibility reifyOnSite
//<
editSavedFromXml : function (xml) {
    if (!xml) return;

    if (this.knownDataSources && !this.session.dataSources) {
        this.session.setDataSources(this.knownDataSources);
    }

    const self = this;
    this.session.dataSourceXmlToDefaults(xml, function (defaults) {
        self.session._addToDataSources({
            ID:defaults.ID,
            defaults:defaults
        });
        const liveDS = self.session.get(defaults.ID);
        self._editSaved(liveDS, defaults);
    });
},

_editSaved : function (dataSource, defaults) {
    if (!isc.isA.MockDataSource(dataSource)) {
        
        if (this.builder && !this.builder.onSiteMode) this.addTestDataButton.show();
        else this.addTestDataButton.hide();
        this.editWithFieldsButton.hide();
    } else {
        this.addTestDataButton.hide();
        if (!this.editSampleData && !dataSource.hasExplicitFields()) {
            this.editWithFieldsButton.show();
        } else {
            this.editWithFieldsButton.hide();
        }
    }

    // When loading sample data it make take considerable time to process include from
    // fields and/or populate the grid. A prompt need to be shown to inform the user
    // what is happening. However, at this point it is normal for the DataSourceEditor to
    // not yet be shown. A showPrompt() call would immediately be hidden when the editor
    // is shown. Instead, to allow the DS Editor to show quickly a flag is set so the
    // binding can begin when the editor is shown.
    if (this.editSampleData) {
        this._bindInitialSampleData(dataSource);
    }
    this.start(dataSource, defaults, false);
},

_bindInitialSampleData : function (dataSource) {
    var dataSource = dataSource || this.getCurrentDataSource();
    if (!dataSource) {
        this.delayCall("_bindInitialSampleData", arguments, 250);
        return;
    };

    var fieldNames = dataSource.getFieldNames(),
        fields = []
    ;
    for (var i = 0; i < fieldNames.length; i++) {
        fields[i] = dataSource.getField(fieldNames[i]);
    }
    this.originalSampleDataOrigin = dataSource.ID;
    this.originalSampleData = dataSource.cacheData;
    // delayed so a showPrompt() has a chance to draw. Rebinding is CPU intensive and
    // does not allow a thread switch so nothing will happen in the UI until it completes.
    this.delayCall("_rebindSampleDataGrid", [fields, dataSource.cacheData, true]);
},

visibilityChanged : function (isVisible) {
    // When editing an existing DS with sample data we don't start processing it until
    // the DS Editor window is shown so the prompt remains visible.
    if (this._pendingRebindSampleData && isVisible) {
        this._bindInitialSampleData();
        delete this._pendingRebindSampleData;
    }
},

//> @method dataSourceEditor.showRecord
// When editor is +link{editSampleData,showing sample data} show a specific sample
// record using the record's primary key.
//
// @param record (Record) the record to be shown
// @visibility reifyOnSite
//<
showRecord : function (record) {
    if (this.editSampleData) {
        // This method is called by Reify immediately after showing the editor to
        // show a specific sample data record, however, it is likely that the sample
        // data grid isn't yet ready because it is updated on delay. Wait for the
        // grid to be set with a DataSource which is done immediately before data
        // is assigned.
        var grid = this.dataGrid;
        if (!grid || !grid.getDataSource()) {
            this.delayCall("showRecord", arguments, 250);
            return;
        }

        // Find matching record by PK because the record is from another source
        var pkFieldName = grid.getDataSource().getPrimaryKeyFieldName(),
            pkValue = record[pkFieldName]
        ;
        this.mainTabSet.selectTab("sampleData");
        var editRows = grid.getAllEditRows();
        for (var i = 0; i < editRows.length; i++) {
            var rowNum = editRows[i],
                editRecord = grid.getEditedRecord(rowNum)
            ;
            if (editRecord[pkFieldName] == pkValue) {
                grid.scrollToRow(rowNum);
                grid.startEditing(rowNum);
                break;
            }
        }
    }
},

start : function (dataSource, defaults, isNew) {
    if (this.editSampleData && !this.mainTabSet.getTabPane(0)) {
        this.mainTabSet.setTabPane(0, this.mainStack);
        this.mainTabSet.show();
    } else if (!this.editSampleData && this.mainTabSet.getTabPane(0)) {
        this.mainTabSet.hide();
        this.addMember(this.mainStack, 0);
    }
    if (this.canEditMockData(dataSource)) {
        this.mainStack.hideSection(1);
        this.mainStack.hideSection(2);
        if (this.showOperationBindingsEditor) this.mainStack.hideSection("operationBindings");
        this.mainStack.showSection(4);
        this.mockEditor.show();
        this._editingMockData = true;
    } else {
        this.mainStack.showSection(1);
        this.mainStack.showSection(2);
        if (this.showOperationBindingsEditor) this.mainStack.showSection("operationBindings");
        this.mainStack.hideSection(4);
        this.mockEditor.hide();
        this._editingMockData = false;
    }

    if (this.mainEditor) this.mainEditor.clearValues();
    if (this.fieldEditor) {
        // While waiting on fields to populate list show helpful message
        this.fieldEditor.grid.setEmptyMessage("Loading DataSource definition... ${loadingImage}");
        this.fieldEditor.setData(null);
    }
    if (this.opBindingsEditor) this.opBindingsEditor.setData(null);

    // Setup edit state for this DataSource (new or existing)
    this._editingNewDataSource = isNew;
    // _origDSName will track the ID of the datasource we were initialized with.
    // This reflects the dataSource ID as it is in storage (if this is an existing DS). Used for
    // rollback and renaming in permanent storage
    // _currentDSName will reflect the dataSource ID currently in the editSession. Used to 
    // detect edits and update the edit session appropriately
    this._origDSName = null;            // set in _start()
    this._currentDSName = null;

    this._origDSDefaults = defaults;

    this._start(dataSource, defaults, isNew);
},

_start : function (dataSource, defaults, isNew) {
    // Make sure editor session has loaded all known DataSource and is ready for use
    if (!this.session.isReady()) {
        var self = this;
        this.session.waitForReady(function () {
            self._start(dataSource, defaults, isNew);
        });
        return;
    }

    if (!dataSource) {
        // no initial dataSource properties at all, start editing from scratch 
        return this.show(); 
    }

    this.dsClass = dataSource.Class;
    if (isNew) {
        // dataSource has never been saved
        if (isc.isA.DataSource(dataSource)) {
            // serializeableFields picks up the fields data - also pick up the
            // sfName if it's defined
            var sfName = dataSource.sfName;
            // currently used only for web service / SalesForce pathways, where we
            // dynamically retrieve a DataSource generated from XML schema.
            dataSource = dataSource.getSerializeableFields();
            if (sfName) dataSource.sfName = sfName;
            
            this.logWarn("editing new DataSource from live DS, data: " + 
                         this.echo(dataSource));
            this._currentDSName = dataSource.ID;
            this._origDSName = this._getOriginalDSName(this._currentDSName);
            this._startEditing(dataSource);
        } else if (!defaults || !defaults.ID) {
            defaults = defaults || {};
            var _this = this;
            if (!defaults.ID) {
                this.getUniqueDataSourceID(function (dsName) {
                    defaults.ID = dsName;
                    _this._currentDSName = dsName;
                    _this._origDSName = _this._getOriginalDSName(_this._currentDSName);
                    _this._startEditing(defaults);
                });
            } else {
                _this.startEditing(defaults);
            }
        } else {
            this._currentDSName = defaults.ID;
            this._origDSName = this._getOriginalDSName(this._currentDSName);
            this._startEditing(defaults);
        }
    } else {
        // dataSource already exists in dsDataSource (i.e. editSaved() called)
        if (defaults) {
            // Save original DS Name to know if it is changed so uniqueness can be checked
            this._currentDSName = defaults.ID;
            this._origDSName = this._getOriginalDSName(this._currentDSName);
            this.delayCall("_startEditing", [defaults]);
        } else {
            // Save original DS Name to know if it is changed so uniqueness can be checked
            this._currentDSName = dataSource.ID;
            this._origDSName = this._getOriginalDSName(this._currentDSName);

            // We need the clean initialization data for this DataSource (the live data
            // contains various derived state)
            var self = this;
            this.session.loadDataSourceDefaults(dataSource.ID, function (defaults) {
                if (!defaults) {
                    self.logWarn("Unable to load defaults for DataSource: " + dataSource.ID);
                    isc.warn("Unable to load defaults for DataSource: " + dataSource.ID, function () {
                        // Without loading a DataSource to edit, the DataSourceEditor is not useful.
                        // Just close it.
                        self.cancelClick();
                    });
                    return;
                }
                self._startEditing(defaults);
            });
        }
    }
},

_getOriginalDSName : function (currentDSName) {
    var origDSName = currentDSName;
    var renamedDS = this.session && this.session.getRenamedDataSources();
    if (renamedDS != null) {
        for (var origName in renamedDS) {
            if (origName[renamedDS] == currentDSName) {
                origDSName = renamedDS;
                break;
            }
        }
    }
   return origDSName;
},

canEditMockData : function (dataSource) {
    return (isc.isA.MockDataSource(dataSource) &&
            this.editMockData != false &&
            !dataSource.hasExplicitFields());
},

switchToEditFieldsAndDataSeparatelyMessage: "By editing fields and data separately " + 
    "additional behaviors can be added to your DataSource, such as validators.<P>" +
    "Once you save in this mode your DataSource will always be edited in this way.<P>" +
    "You can always re-create your DataSource from sample data. You may want to take " +
    "a copy of your current sample data before you begin editing.",

switchToEditFieldsAndDataSeparately : function () {
    var _this = this;
    isc.warn(this.switchToEditFieldsAndDataSeparatelyMessage, function (response) {
        if (response) _this._switchToEditFieldsAndDataSeparately();
    }, {
        buttons: [isc.Dialog.CANCEL, isc.Dialog.OK],
        autoFocusButton: 1
    });
},

_switchToEditFieldsAndDataSeparately : function () {
    // Push any changes to mock data into session
    var defaults = this.pushEditsToSession();

    this.editSampleData = true;

    // When editing sample data put main editor into a tab and add another tab for data
    this.mainTabSet = this.createAutoChild("mainTabSet");

    // Sample Data tab contents
    var label = this.createAutoChild("sampleDataLabel");
    this.dataGrid = this.createAutoChild("sampleDataGrid");

    var addNewButton = this.createAutoChild("sampleDataAddRecordButton");
    var discardDataButton = this.createAutoChild("sampleDataDiscardDataButton");
    var buttonLayout = this.createAutoChild("sampleDataButtonLayout", {
        members: [ addNewButton, discardDataButton ]
    });

    this.dataPane = this.createAutoChild("sampleDataPane", {
        members: [ label, this.dataGrid, buttonLayout ] });

    // Create tabs
    this.mainTabSet.addTab({
        name: "fields",
        title: "DataSource Fields",
        pane: this.mainStack
    });
    this.mainTabSet.addTab({
        name: "sampleData",
        title: "Sample Data",
        pane: this.dataPane
    });

    this.addMember(this.mainTabSet, 0);

    // Update mainStack to show DS and field editors
    this.mainStack.showSection(1);
    this.mainStack.showSection(2);
    this.mainStack.showSection(3);
    this.mainStack.hideSection(4);
    this.mockEditor.hide();
    this._editingMockData = false;

    // Convert to standard DataSource
    var defaults = this.session.standardizeMockDataSource(defaults.ID);

    // Start editing again with updated defaults
    this._startEditing(defaults, true);

    // Bind the sample data editor
    this.originalSampleData = defaults.cacheData;
    this._rebindSampleDataGrid(defaults.fields, defaults.cacheData);

    // Don't need the button change editor anymore
    this.editWithFieldsButton.hide();
},

//> @method dataSourceEditor.getUniqueDataSourceID
// Called to obtain a unique DataSource ID for a new DataSource.
// <P>
// By default this method uses 'newDataSource' unless a dataSource with that
// name already exists in +link{knownDataSources,this.knownDataSources}. If
// 'newDataSource' is already taken, it will append a numeric suffix 
// ('newDataSource0' if free, otherwise, 'newDataSource1', 'newDataSource2', or whatever
// the first unused ID is).
// <P>
// Override to implement a custom algorithm.
//
// @param callback (Callback) Callback to fire when the unique ID is determined. Takes a single
//                            parameter <code>ID</code> - the unique DataSource ID
// @visibility reifyOnSite
//<
getUniqueDataSourceID : function (callback) {
    var baseName = "newDataSource";
    var uniqueName = baseName;
    if (this.knownDataSources && (this.knownDataSources.findIndex("ID",baseName) != -1)) {
        var suffix = 0;
        uniqueName = baseName+suffix;
        var dsIDs = this.knownDataSources.getProperty("ID");
        while (dsIDs.indexOf(uniqueName) >=0) {
            suffix++;
            uniqueName = baseName + suffix;
        }
    }
    callback(uniqueName);
},

_startEditing : function (defaults, reloaded) {
    if (this.mainEditor) this.mainEditor.setValues(defaults);
    else this.mainEditorValues = defaults;

    if (!this._editingMockData) {
        // Bind fields to FieldEditor
        this.bindFields(defaults);
    }

    if (this.opBindingsEditor) {
        if (defaults) {
            var bindings = defaults.operationBindings;
            if (bindings && bindings.length > 0) {
                // Extract code bindings and create data to edit
                var values = {};
                var opTypes = ["fetch","add","update","remove"];
                for (var i = 0; i < bindings.length; i++) {
                    var binding = bindings[i];
                    if (binding.operationType && opTypes.contains(binding.operationType) &&
                        !binding.operationId)
                    {
                        var roles = binding.requiresRole;
                        if (binding.requires == "false") {
                            roles = "false";
                        }
                        values[binding.operationType + "RequiresRole"] = roles;
                    }
                }
                this.opBindingsEditor.setValues(values);
            }
        }
    }
    if (this.mockEditor) {
        this.mockEditor.setValue("ID", defaults.ID);
        if (defaults.mockData) {
            var mockData = defaults.mockData;
            if (!defaults.mockDataFormat || defaults.mockDataFormat == "mock") {
                mockData = mockData.replace(/\\/g, "\\").replace(/^\[(.*)\]$/m, "{$1}");
            }
            this.mockEditor.setValue("edit", mockData);
            this.mockEditor.setValue("mockDataFormat", defaults.mockDataFormat);
            if (defaults.mockDataPrimaryKey) {
                this.mockEditor.setValue( "addField", null );
                this.mockEditor.setValue( "useField", true );
                this.mockEditor.setValue( "fieldName", defaults.mockDataPrimaryKey );
            } else {
                this.mockEditor.setValue( "addField", true );
                this.mockEditor.setValue( "useField", null );
            }
            this.mockEditor.editDataChanged(mockData);
        }
    }

    if (!reloaded) {
        // Save initial DS configuration as represented in the editors. It can then be compared
        // later to determine if any changes have been made that need to be saved. See hasChanges()
        this._origDSDefaults = this.getDefaults();

        // If the DS has sample data the _origDSDefaults may not have corresponding cacheData yet.
        // We cannot just use the defaults.cacheData because it won't have been converted to the
        // same data types used by the sample editor. Additionally, we cannot wait until after the
        // _rebindSampleDataGrid() call because that triggers the grid update on another thread.
        // Rather, set a flag to have the rebinding update the _origDSDefaults upon completion.
        if (defaults && defaults.cacheData && !this._origDSDefaults.cacheData) {
            this._updateOrigDSDefaultsOnSampleDataBinding = true;
        }

        // Save original defaults into session so they can be compared for change reporting
        this.session.add(defaults.ID, this._origDSDefaults, !this._editingNewDataSource);
    }

    // Update editor state
    this.fieldEditor.updateButtonStates();
    this.fieldEditor.updateIncludeFieldButtonState();
    this.updateTitleField();

    // Set flag to indicate that editor is completely ready for use.
    // This is used by save() to prevent an attempted save before. This can occur
    // when this editor is used, without showing to the end user, to save a new
    // DataSource.
    this._ready = true;
},

bindFields : function (defaults) {
    // Re-bind defaults to FieldEditor
    var fieldEditor = this.fieldEditor;
    if (fieldEditor) {
        var selectedRecord = fieldEditor.grid.getSelectedRecord();
        var allFields = this.getAllFields(defaults);

        if (this.autoAddPK && !this.hasPrimaryKeyField(allFields)) {
            this.createUniqueIdField(allFields);
        }
        // We now have data to populate the fields; reset the empty message
        fieldEditor.grid.setEmptyMessage("No fields to show.");
        if (this.canEditChildSchema) {
            this.setupIDs(allFields, 1, null);

            var tree = isc.Tree.create({
                modelType: "parent",
                childrenProperty: "fields",
                titleProperty: "name",
                idField: "id",
                nameProperty: "id",
                root: { id: 0, name: "root"},
                data: allFields
            });
            tree.openAll();
            fieldEditor.setData(tree);
        } else {
            fieldEditor.setData(allFields);
        }
        if (this.canSelectPrimaryKey) {
            fieldEditor.grid.showField("primaryKey");
        } else {
            fieldEditor.grid.hideField("primaryKey");
            fieldEditor.grid.getField("primaryKey").canHide = false;
        }
        if (selectedRecord) {
            fieldEditor.grid.selectSingleRecord(selectedRecord);
        }
        fieldEditor.targetDataSource = defaults;
        fieldEditor.formLayout.hide();
        fieldEditor.gridLayout.show();
    }
},

setupIDs : function (fields, nextId, parentId) {
    var index=nextId,
        item,
        subItem
    ;

    if (!index) index = 1;
    for (var i = 0; i < fields.length; i++) {
        var item = fields.get(i);
        item.parentId = parentId;
        item.id = index++;
        if (item.fields) {
            if (!isc.isAn.Array(item.fields)) item.fields = isc.getValues(item.fields);
            index = this.setupIDs(item.fields, index, item.id);
        }
    }
    return index;
},

getAllFields : function (defaults) {
    var myFields = (defaults && defaults.fields) || [];
    if (!isc.isAn.Array(myFields)) myFields = isc.getValues(myFields);

    var fields = isc.clone(myFields);

    if (this.showInheritedFields) {
        var parentFields = this.getParentFields(defaults);
        if (parentFields && fields) {
            // Place parent fields before my fields
            fields.addListAt(parentFields, 0);
        } else if (parentFields) {
            // No local fields
            fields = parentFields;
        }
    }

    return fields;
},

getParentFields : function (defaults) {
    var fields = [];
    if (defaults.inheritsFrom) {
        var parentDefaults = this.session.getDataSourceDefaults(defaults.inheritsFrom);
        if (parentDefaults) {
            if (parentDefaults.fields) {
                fields = isc.clone(parentDefaults.fields);
                fields.setProperty("_inheritedFrom", defaults.inheritsFrom);
            }
            // Place parent fields before previous fields
            fields.addListAt(this.getParentFields(parentDefaults), 0);
        }
    }
    return fields;
},

getFields : function (localOnly) {
    var fields;
    if (this.canEditChildSchema) {
        var tree = this.fieldEditor.grid.data;
        fields = tree.getCleanNodeData(tree.getRoot(), true).fields;
        fields = this.getExtraCleanNodeData(fields);
    } else { 
        fields = isc.clone(this.fieldEditor.getData());
    }
    // Remove any records that are pending a removal from the returned list
    var recordsToRemove = fields.findAll("_removing", true);
    if (recordsToRemove) fields.removeList(recordsToRemove);

    // Remove any inherited records from the returned list
    if (localOnly && this.showInheritedFields) {
        var recordsToRemove = fields.filter(function (record) {
            return (record._inheritedFrom != null);
        });
        if (recordsToRemove) fields.removeList(recordsToRemove);
    }
    // Drop property identifying the record as inherited
    fields.map(function (record) {
        delete record._inheritedFrom;
    });
    
    // Empty fields can be populated in the record causing the generated field in the DS
    // to have values like title="" which suppresses desired auto-title generation.
    if (fields != null) {
        for (var i = 0; i < fields.length; i++) {
            var field = fields[i];
            for (var key in field) {
                if (field[key] == null || field[key] === "") delete field[key];
            }
        }
    }
    return fields;
},

// return the field type from the record taking into account a possible includeFrom value
getFieldType : function (record) {
    var type = record.type;
    if (record.includeFrom) {
        var split = record.includeFrom.split(".");
        if (split && split.length >= 2) {
            var dsName = split[split.length-2],
                dsField = split[split.length-1],
                ds = this.session.get(dsName)
            ;
            if (ds) {
                var field = ds.getField(dsField);
                if (field) {
                    type = field.type;
                }
            }
        }
    }
    return isc.SimpleType.getBaseType(type);
},

// Apply operation binding changes to ds defaults
applyOpBindings : function (dsData) {
    
    // If the operation bindings editor is suppressed, nothing to do
    if (!this.opBindingsEditor) return;

    var editorData = this.opBindingsEditor.getValues();

    var bindings = {};
    ["fetch","add","update","remove"].map(function (operationType) {
        var requiresRole = editorData[operationType + "RequiresRole"],
            binding = null
        ;
        // Process special values
        if (requiresRole == "false") {
            binding = { requires: "false" };
        } else if (requiresRole && requiresRole != "_any_") {
            // Convert array of selections to a comma-separated string
            if (isc.isAn.Array(requiresRole)) {
                requiresRole = requiresRole.join(",");
            }
            binding = { requiresRole: requiresRole };
        }
        if (binding) bindings[operationType] = binding;
    });

    if (!dsData.operationBindings) dsData.operationBindings = [];

    var oldBindings = dsData.operationBindings;
    ["fetch","add","update","remove"].map(function (operationType) {
        var oldBinding = null,
            matchingBindings = oldBindings.findAll("operationType", operationType)
        ;
        // Only apply changes to matching binding with no operationId
        if (matchingBindings) {
            for (var i = 0; i < matchingBindings.length; i++) {
                if (!matchingBindings[i].operationId) {
                    oldBinding = matchingBindings[i];
                    break;
                }
            }
        }

        var newBinding = bindings[operationType];
        if (!oldBinding && !newBinding) return;

        if (oldBinding && !newBinding) {
            if (oldBinding.requires) delete oldBinding.requires;
            if (oldBinding.requiresRole) delete oldBinding.requiresRole;
            // If only the "operationType" property remains, drop the binding completely
            if (isc.getKeys(oldBinding).length == 1) dsData.operationBindings.remove(oldBinding);
        } else if (!oldBinding && newBinding) {
            newBinding.operationType = operationType;
            oldBindings.add(newBinding);
        } else {
            if (oldBinding.requires && !newBinding.requires) delete oldBinding.requires;
            else if (newBinding.requires) oldBinding.requires = newBinding.requires;
            if (oldBinding.requiresRole && !newBinding.requiresRole) delete oldBinding.requiresRole;
            else if (newBinding.requiresRole) oldBinding.requiresRole = newBinding.requiresRole;
            // If only the "operationType" property remains, drop the binding completely
            if (isc.getKeys(oldBinding).length == 1) dsData.operationBindings.remove(oldBinding);
        }
    });

    // Make sure that when there are no operation binding, nothing is serialized
    if (dsData.operationBindings.length == 0) delete dsData.operationBindings;
},

getDataSourceID : function () {
    var ID = (this.mainEditor ? this.mainEditor.getValue("ID") : null);
    if (this.dsClass == "MockDataSource" && this._editingMockData) {
        ID = this.mockEditor.getValue("ID");
    }
    return ID;
},

getDatasourceData : function () {
    // NOTE: dsClass is set when we begin editing
    var dsClass = this.dsClass || "DataSource",
        dsData = isc.addProperties({}, 
            this.mainEditor ? this.mainEditor.getValues() : this.mainEditorValues
        )
    ;

    if (!this._editingMockData) {
        // Get all local fields from FieldEditor
        dsData.fields = this.getFields(true);
    }

    this.applyOpBindings(dsData);

    // When editing sample data pull current values and use as cacheData
    if (this.editSampleData) {
        var sampleData = this.getSampleData();
        dsData.cacheData = sampleData;
    } else if (dsClass == "MockDataSource" && this._editingMockData) {
        var ID = this.mockEditor.getValue("ID"),
            mockData = this.mockEditor.getValue("edit") || "",
            mockDataType = dsData.mockDataType || "grid",
            mockDataFormat = dsData.mockDataFormat || "mock",
            primaryKeySpecified = this.mockEditor.getValue("useField"), 
            mockDataPrimaryKey = primaryKeySpecified && this.mockEditor.getValue("fieldName")
        ;
        dsData.ID = ID;
        dsData.mockData = (mockDataFormat != "mock" || mockDataType == "tree" ? mockData.trim() : 
            mockData.trim().replace(/\\/g, "\\").replace("{", "[").replace("}", "]"));
        if (mockDataPrimaryKey) {
            dsData.mockDataPrimaryKey = mockDataPrimaryKey;
        } else {
            delete dsData.mockDataPrimaryKey;
        }
        dsData.fromServer = true;
        // These properties are derived from the mockData on initialization
        delete dsData.cacheData;
        delete dsData.fields;
    }

    return dsData;
},

hasPrimaryKeyField : function (fields) {
    if (!fields) fields = this.getFields();

    // Determine if there is a defined field marked as PK or
    // DataSource inherits a PK field.
    var hasPK = fields.getProperty("primaryKey").or();
    if (!hasPK) {
        // This DataSource might inherit its primaryKey field...
        var ds = this.getCurrentDataSource();
        if (ds) {
            var allFields = ds.getFields();
            for (var key in allFields) {
                var fld = allFields[key];
                // Catch the case that the user has overridden its inherited PK
                // field and removed the primaryKey designation
                if (fld.primaryKey && !fields.find("name",key)) {
                    hasPK = true;
                    break;
                }
            }
        }
    }
    return hasPK;
},

// DS ID changed in the main editor
dataSourceIDChanged : function (origID, newID) {
    // Update session with changes. This handles the rename so we have functional
    // DataSource to work with.
    this.pushEditsToSession();

    var changeContext = this.dataSourcePropertyChanged("ID", origID, newID);

    // If an automatic PK was assigned, update it with a name matching the new DS ID
    if (this.autoAddPK) {
        var fields = this.getFields();
        if (fields) {
            var uniqueIdField = fields.find("name", this.uniqueIdFieldName);
            if (!uniqueIdField) uniqueIdField = fields.find("name", "internalId");
            if (uniqueIdField && uniqueIdField.primaryKey) {
                var pkName = newID + "Id";
                if (!fields.find("name", pkName)) {
                    var editor = this.fieldEditor,
                        grid = editor.grid,
                        totalRows = grid.getTotalRows()
                    ;
                    this.session.startChangeContext(changeContext);
                    for (var rowNum = 0; rowNum < totalRows; rowNum++) {
                        var record = grid.getRecord(rowNum);
                        if (record && record.name == uniqueIdField.name) {
                            var colNum = grid.getFieldNum("name"),
                                alreadyEditing = (grid.getEditRow() == rowNum)
                            ;
                            if (!alreadyEditing) grid.startEditing(rowNum, colNum);
                            grid.setEditValue(rowNum, colNum, pkName);
                            if (!alreadyEditing) grid.endEditing();
                            break;
                        }
                    }
                    this.session.endChangeContext(changeContext);
                }
            }
        }
    }
},

dataSourcePropertyChanged : function (property, oldValue, newValue) {
    var dsName = this.mainEditor.getValue("ID");
    return this.session.addChange(dsName, "changeProperty", {
        property: property,
        originalValue: oldValue,
        newValue: newValue
    });
},

updateTitleField : function (fromName, toName) {
    if (this._editingMockData) {
        return;
    }
    var titleField = this.mainEditor.getField("titleField");
    if (!titleField) return;

    var fieldsDS = this.getFieldsDS(this.getFields()),
        dsId = this.mainEditor.getValue("ID"),
        fieldsDSId = fieldsDS.ID
    ;
    // The fieldsDS is a temporary DS generated from the current edit fields so the ID
    // is not the same as the user-specified ID. Since determining the titleField will check
    // for fields prefixed with the ID we temporarily set the DS ID to the desired ID
    // while pulling the titleField value.
    fieldsDS.ID = dsId;

    var defaultTitleField = fieldsDS.getTitleField(),
        emptyDisplayValue = (defaultTitleField ? "default: <i>" + defaultTitleField + "</i>" : "[None]"),
        fieldNames = fieldsDS.getFieldNames(),
        valueMap = []
    ;
    fieldsDS.ID = fieldsDSId;

    for (var i = 0; i < fieldNames.length; i++) {
        var field = fieldsDS.getField(fieldNames[i]);
        if (!field.type || field.type == "text" || field.type == "enum") {
            valueMap.add(field.name);
        }
    }
    valueMap.sort();

    var value = titleField.getValue();
    if (value && !valueMap.contains(value)) {
        if (value == fromName) {
            // Selected titleField value points to renamed field. Update it.
            titleField.setValue(toName);
        } else {
            // Target field no longer exists. Remove reference.
            titleField.clearValue();
        }
    }
    titleField.emptyDisplayValue = emptyDisplayValue;
    titleField.setValueMap(valueMap);
},

getFieldsDS : function (fields) {
    // If showing sample data the testDS is assumed to already be up-to-date
    if (this.testDS) return this.testDS;

    // Create temporary "fieldsDS" for use by updateTitleField to get default titleField
    if (this._fieldsDS) this._fieldsDS.destroy();

    var dsProperties = isc.addProperties({}, this.dsProperties, {
        clientOnly: true
    });
    if (fields) {
        dsProperties.fields = isc.clone(fields);

        // Add a primaryKey "internalId" field if there isn't one already.
        // Do this after cloning the fields - don't impact the fields object passed in
        var dsPropFields = dsProperties.fields;
        if (!dsPropFields.find("primaryKey", true)) {
            dsPropFields.addAt({ name: "internalId", type: "sequence", primaryKey: true, hidden: true }, 0);
        }
        
    }

    this._fieldsDS = isc.DataSource.create(dsProperties);

    return this._fieldsDS;
},

// ---------------------------------------------------------------------------------------
// Sample data editor support

getSampleData : function () {
    var rowCount = this.dataGrid.getTotalRows();
    if (rowCount == 0) return null;

    var grid = this.dataGrid,
        data = [],
        // _operation + [pk]
        baseFieldsInRecord = 1 + (grid._pkFieldName ? 1 : 0)
    ;
    for (var i = 0, row = 0; i < rowCount; i++) {
        var record = grid.getEditValues(i);
        if (!grid.recordMarkedAsRemoved(i)) {
            // Only save records that have some field values other than the PK
            if (!isc.isAn.emptyObject(record)) {
                var isAddOp = (record._operation == "add"),
                    found = 0;
                for (var key in record) {
                    if (((isAddOp && record[key] != null && record[key] != "") || !isAddOp) &&
                        ++found > baseFieldsInRecord)
                    {
                        data[row++] = record;
                        break;
                    }
                }
            }
        } else {
            record._operation = 'remove';
            data[row++] = record;
        }
    }
    return data;
},

renameSampleDataField : function (fromName, toName, data) {
    data = data || this.getSampleData();

    // Rename field in records
    if (data) {
        data.forEach(function (record) {
            if (record[fromName] != null) {
                record[toName] = record[fromName];
                delete record[fromName];
            }
        });
    }

    return data;
},

revertDataChanges : function () {
    this._setDataGridData(this.originalSampleData);
},

// Rebind Sample Data using existing fields and data
rebindSampleData : function () {
    if (this.editSampleData) {
        var fields = this.getFields(),
            data = this.getSampleData()
        ;
        // Re-create testDS with updated fields/data
        this._rebindSampleDataGrid(fields, data);
    }
},

_rebindSampleDataGrid : function (fields, records, initialData) {
    if (this.isDrawn() && this.isVisible()) {
        // A ${loadingImage} isn't helpful because the UI is frozen during the process
        isc.showPrompt((initialData ? "Loading" : "Updating") + " sample data...");
    }
    this.delayCall("__rebindSampleDataGrid", arguments);
},

__rebindSampleDataGrid : function (fields, records) {
    // Re-create testDS with updated fields/data
    if (this.testDS) this.testDS.destroy();

    var _this = this;
    var dsProperties = isc.addProperties({}, this.dsProperties, {
        clientOnly: true,
        // For FK fields that reference the same DS (tree) the DS doesn't actually have
        // data to filter for display in the picklist because the grid has all records in
        // edit mode. On fetch response, populate the data with the sample data in the
        // grid edit records.
        transformResponse : function (dsResponse, dsRequest, data) {
            if (dsResponse.operationType == "fetch") {
                var gridData = _this.getSampleData();
                dsResponse.startRow = dsRequest.startRow || 0;
                dsResponse.endRow = Math.min(gridData.length, dsRequest.endRow || 0);
                dsResponse.totalRows = dsResponse.endRow - dsResponse.startRow;
                dsResponse.data = gridData.slice(dsRequest.startRow, dsResponse.endRow);
            }
            return dsResponse;
        }
    });
    if (fields) {
        // Need a PK so editing can match up records without logging tons of warnings
        if (!fields.find("primaryKey", true)) {
            fields.addAt({ name: "internalId", type: "sequence", primaryKey: true, hidden: true }, 0);
        }

        // A field validator might have a dynamicErrorMessageArguments applied because it
        // has been used. Within that property is a reference to its validator which will
        // cause the clone() below to loop. Drop any dynamicErrorMessageArguments validator
        // property because it gets re-created on every evaluation.
        //
        // Additionally, Regexp properties like 'mask' and 'expression' will not copy
        // correctly leading to a broken validator. Restore these properties to their
        // original string values for copying.
        for (var i = 0; i < fields.length; i++) {
            var field = fields[i],
                validators = field.validators || [];
            for (var j = 0; j < validators.length; j++) {
                var validator = validators[j];
                if (validator.type == "regexp" && !isc.isA.String(validator.expression)) {
                    validator.expression = validator.expression.source;
                } else if (validator.type == "mask" && !isc.isA.String(validator.mask)) {
                    validator.mask = validator.mask.source;
                }
                delete validator.dynamicErrorMessageArguments;
            }
        }

        dsProperties.fields = isc.clone(fields);
    }

    this.testDS = isc.DataSource.create(dsProperties);

    // Rebind grid
    this.dataGrid.setDataSource(this.testDS);

    // Apply current data
    this._setDataGridData(records, this.testDS.getID());

    // On initial edit, update the _origDSDefaults cacheData with the sample data
    if (this._updateOrigDSDefaultsOnSampleDataBinding) {
        delete this._updateOrigDSDefaultsOnSampleDataBinding;
        var defaults = this.getDefaults();
        this._origDSDefaults = defaults;
        // push fully updated defaults into session
        this.session.set(defaults.ID, defaults, true);
    }

    isc.clearPrompt();
},

_setDataGridData : function (records, dsID) {
    var grid = this.dataGrid;

    // if we can't resolve includeFrom field values during a fetch, we'll do it manually
    // using the provided records.  We'll pass the name of the test datasource ID
    // when we do so, in case its needed 
    if (records != null) {
        grid.dataSource.resolveClientOnlyIncludeFrom(records, dsID);
    }

    // If the DS has a PK - it does because an internal one is added if the user has not
    // defined an explicit one - and it is a sequence initialize state fields on the grid
    // so new sequence values can be assigned as needed. This would normally be handled
    // by the DS but since records are never saved to the DS that process does not occur.
    var pkField = grid.getDataSource().getPrimaryKeyField();
    if (pkField && pkField.type == "sequence") {
        grid._pkFieldName = pkField.name;
        grid._pkNextValue = 1;
    } else {
        delete grid._pkFieldName;
        delete grid._pkNextValue;
    }

    grid.discardAllEdits();
    if (records && records.length > 0) {
        // Determine the highest sequence PK value in the current data so new records can
        // be assigned values above that.
        var pkFieldName = grid._pkFieldName,
            pkFieldMaxValue = -1
        ;
        if (pkFieldName) {
            for (var i = 0; i < records.length; i++) {
                var record = records[i],
                    pkValue = record[pkFieldName]
                ;
                if (pkValue != null) {
                    
                    if (!isc.isA.String(pkValue) && pkValue > pkFieldMaxValue) {
                        pkFieldMaxValue = pkValue;
                    }
                }
            }
            grid._pkNextValue = pkFieldMaxValue+1;
        }
        // Place records into the grid as edit values so validation errors show
        for (var i = 0; i < records.length; i++) {
            var record = records[i];
            // Assign sequence PK value if missing
            if (pkFieldName && record[pkFieldName] == null) {
                record[pkFieldName] = grid._pkNextValue++;
            }
            grid.startEditingNew(record, true);
        }
        // Add one more record leaving the real records pending
        grid.startEditing(0);
    } else {
        grid.startEditingNew();
    }
},

// ---------------------------------------------------------------------------------------
// Field property pop-up editors

legalValuesWindowDefaults: {
    _constructor: isc.Window,
    autoCenter: true,

    height: 250,
    width: 500,

    isModal: true,
    showModalMask: true,
    showHeaderIcon: false,
    showMinimizeButton: false,
    keepInParentRect: true,
    close : function () {
        this.Super("close", arguments);
        this.markForDestroy();
    }
},

legalValuesFormDefaults: {
    _constructor: isc.DynamicForm,
    addAsChild: true,
    width: "100%",
    height: "100%",
    numCols: 1,
    fields: [
        { name: "values", editorType: "ValueMapItem",
            showTitle: false, showMapTypeButton: false, showHeader: false,
            newOptionRowMessage: "Click to add allowed values to the list" }
    ],
    // Override to force validation on ValueMapItem so it can push pending changes to form values
    validate : function () {
        this.getItem("values").validate();
        return this.Super("validate", arguments);
    }
},

legalValuesToolbarDefaults: {
    _constructor: isc.HLayout,
    width: "100%",
    height: 30,
    padding: 10,
    align: "right",
    membersMargin: 4,
    members: [
        { _constructor: isc.Button,
            title: "Cancel",
            width: 75,
            click: function () {
                this.topElement.destroy();
            }
        },
        { _constructor: isc.Button,
            title: "Save",
            width: 75,
            click: function () {
                this.parentElement.saveLegalValues();
            }
        }
    ]
},

editFieldLegalValues : function (field) {
    var legalValuesWindowProperties = {
        title: "Define the allowed values for " + field.name
    }
    var window = this.createAutoChild("legalValuesWindow", legalValuesWindowProperties);

    var legalValuesFormProperties = {
        values: { values: field.valueMap }
    };
    this.legalValuesForm = this.createAutoChild("legalValuesForm", legalValuesFormProperties);

    var legalValuesToolbarProperties = {
        window : window,
        editor : this.legalValuesForm,
        saveLegalValues : function () {
            if (this.editor.validate()) {
                var valueMap = this.editor.getValue("values");
                field.valueMap = valueMap;
                this.window.markForDestroy();

                this.editor.creator.rebindSampleData();
            }
        }
    };
    this.legalValuesToolbar = this.createAutoChild("legalValuesToolbar", legalValuesToolbarProperties);

    window.addItem(this.legalValuesForm);
    window.addItem(this.legalValuesToolbar);
    window.show();
},

derivedValueWindowDefaults: {
    _constructor: isc.Window,
    autoCenter: true,

    height: 600,
    width: 650,

    isModal: true,
    showModalMask: true,
    showHeaderIcon: false,
    showMinimizeButton: false,
    keepInParentRect: true,
    close : function () {
        this.Super("close", arguments);
        this.markForDestroy();
    }
},

derivedValueEditorConstructor:"DynamicValueEditor",
derivedValueEditorDefaults:{
    getBuilderProperties : function () {
        return {
            dataSource: this.dataSource,
            // Just the math functions supported by server-side formulas
            mathFunctions: ["min", "max", "random", "pow", "ceil", "ln", "year", "month", "day"]
        };
    }
},

editDerivedValue : function (field) {

    var fieldType = (field.type || "string").toLowerCase(),
        valueTypes = [ "textFormula", "formula", "conditional" ],
        fixedType = null,
        itemEditorConstructor = null,
        valueMap = null,
        type = "formula or template"
    ;
    if (fieldType == "integer" || fieldType == "long" ||
        fieldType == "float" || fieldType == "number")
    {
        // A numeric field only supports a formula although it might make sense to
        // allow a conditional
        fixedType = "formula";
        type = "formula";
    } else if (fieldType == "enum") {
        valueTypes = [ "textFormula", "formula", "conditional" ];
        // Conditional is restricted to enum valueMap
        // itemEditorConstructor = "DynamicValueEnumMappingEditor";
        // valueMap = field.valueMap;
        type = "formula or template";
    } else if (fieldType == "date" || fieldType == "datetime" || fieldType == "time") {
        // A date or time field only supports a template
        fixedType = "textFormula";
        type = "template";
    }

    var windowProperties = {
        title: "Define a " + type + " for " + field.name
    }
    var window = this.createAutoChild("derivedValueWindow", windowProperties);

    var editor = this.createAutoChild("derivedValueEditor", {
        dataSource: this.getCurrentDataSource(),
        valueTypes: valueTypes,
        fixedType: fixedType,
        itemEditorConstructor: itemEditorConstructor,
        valueMap: valueMap,

        fireOnClose:function () {
            var dsEditor = editor.creator;
            if (!editor.cancelled) {
                var tree = dsEditor.fieldEditor.grid.data,
                    oldField = (isc.isA.Tree(tree) ? tree.getCleanNodeData(field) : field);
                // Start with no dynamic value to allow clearing
                delete field.formula;
                delete field.template;
                delete field.valueFrom;
                // Add any defined dynamic value
                if (editor.isSimpleFormula()) {
                    field.formula = editor.getSimpleFormula();
                } else if (editor.isSimpleSummary()) {
                    field.template = editor.getSimpleSummary();
                } else if (editor.isConditional()) {
                    field.valueFrom = editor.getConditional();
                }
                dsEditor.saveFieldChanges(["formula","template","valueFrom"], oldField, field)
            }
            window.markForDestroy();
            if (!editor.cancelled) {
                dsEditor.rebindSampleData();
            }
        }
    });

    // Set current value in editor
    if (field.formula) {
        editor.setSimpleFormula(field.formula);
    } else if (field.template) {
        editor.setSimpleSummary(field.template);
    } else if (field.valueFrom) {
        editor.setConditional(field.valueFrom);
    }

    window.addItem(editor);
    window.show();
},

validatorsWindowDefaults: {
    _constructor: isc.Window,
    autoCenter: true,

    height: 550,
    width: 800,

    isModal: true,
    showModalMask: true,
    showHeaderIcon: false,
    showMinimizeButton: false,
    keepInParentRect: true,
    close : function () {
        this.Super("close", arguments);
        this.markForDestroy();
    },
    destroy : function () {
        if (this.dataSource) this.dataSource.destroy();
        this.Super("destroy", arguments);
    }
},

validatorsLayoutDefaults: {
    _constructor: isc.ValidatorsEditor,
    addAsChild: true,
    width: "100%",
    height: "100%"
},

validatorsToolbarDefaults: {
    _constructor: isc.HLayout,
    width: "100%",
    height: 30,
    padding: 10,
    align: "right",
    membersMargin: 10,
    members: [
        { _constructor: isc.IButton,
            title: "Cancel",
            width: 75,
            click: function () {
                this.topElement.destroy();
            }
        },
        { _constructor: isc.IButton,
            title: "Save",
            width: 75,
            click: function () {
                this.parentElement.saveValidators();
            }
        }
    ]
},

editFieldValidators : function (field) {
    // Create a temporary DataSource for use by validatorsEditor.
    // fields array is updated by the DS creation so deep clone
    // it to avoid affecting the edits.
    var dsData = this.getDatasourceData();
    var dsID = dsData.ID;
    delete dsData.ID;
    dsData.fields = isc.clone(dsData.fields);
    var ds = this.createLiveDSInstance(dsData);

    var dsField = dsData.fields.find("name", field.name),
        validators = dsField && dsField.validators || field.validators
    ;

    var validatorsWindowProperties = {
        title: "Validators for " + field.name,
        // Window is responsible for destroying temp DS when closed
        dataSource: ds
    }
    var window = this.createAutoChild("validatorsWindow", validatorsWindowProperties);

    var validatorsLayoutProperties = {
        fieldName: field.name,
        dataSource: ds,
        validators: validators
    };
    this.validatorsLayout = this.createAutoChild("validatorsLayout", validatorsLayoutProperties);

    var validatorsToolbarProperties = {
        window : window,
        editor : this.validatorsLayout,
        saveValidators : function () {
            if (this.editor.validate()) {
                // Get all non-typecast validators
                var validators = this.editor.getValidators(true),
                    oldValidators = field.validators
                ;
                field.validators = validators;
                this.window.markForDestroy();

                this.editor.creator.session.addChange(dsID, "changeFieldProperty", {
                    fieldName: field.name,
                    property: "validators",
                    originalValue: oldValidators,
                    newValue: validators
                });

                this.editor.creator.rebindSampleData();
            }
        }
    };
    this.validatorsToolbar = this.createAutoChild("validatorsToolbar", validatorsToolbarProperties);

    window.addItem(this.validatorsLayout);
    window.addItem(this.validatorsToolbar);
    window.show();
},

securityWindowDefaults: {
    _constructor: isc.Window,

    isModal: true, showModalMask: true,
    showHeaderIcon: false,
    showMinimizeButton: false,
    keepInParentRect: true,
    autoCenter:true,
    autoSize:true,
    canDragResize:true
},

securityEditorDefaults: {
    _constructor: "VLayout",
    autoDraw: false,
    width: 400,
    membersMargin: 10,

    formDefaults: {
        _constructor: "DynamicForm",
        autoDraw: false,
        padding: 10,
        wrapItemTitles: false
    },

    buttonLayoutDefaults: {
        _constructor: "HLayout",
        autoDraw: false,
        width: "100%",
        height:42,
        layoutMargin:10,
        membersMargin:10,
        align: "right"
    },

    //> @attr dataSourceEditor.cancelButton (AutoChild Button : null : R)
    // Cancel button autoChild to be displayed if +link{showActionButtons} is true. Fires
    // +link{cancelClick()} on click.
    // @visibility reifyOnSite
    //<


    cancelButtonDefaults: {
        _constructor: "IButton",
        autoDraw: false,
        title: "Cancel",
        width: 75,
        autoParent: "buttonLayout",
        click: function() {
            this.creator.cancelClick();
        }
    },

    //> @attr dataSourceEditor.saveButton (AutoChild Button : null : R)
    // Save button autoChild to be displayed if +link{showActionButtons} is true. Fires
    // +link{saveClick()} on click.
    // @visibility reifyOnSite
    //<

    saveButtonDefaults: {
        _constructor: "IButton",
        autoDraw: false,
        title: "Save",
        width: 75,
        autoParent: "buttonLayout",
        click: function() {
            this.creator.saveClick();
        }
    },

    initWidget : function () {
        this.Super('initWidget', arguments);

        var fields = [
            { type: "blurb", defaultValue: "Configure the roles required to view or modify this field" },
            { type: "RowSpacer" },
            this.createRolesField("viewRequiresRole", "Roles required to <i>view</i> this field"),
            this.createRolesField("editRequiresRole", "Roles required to <i>edit</i> this field")
        ];

        this.addAutoChild("form", { fields: fields });
        this.addAutoChild("buttonLayout");
        this.buttonLayout.addMember(this.createAutoChild("cancelButton"));
        this.buttonLayout.addMember(this.createAutoChild("saveButton"));
    },

    specialFieldValues: ["_any_","*super*","false"],

    createRolesField : function (fieldName, title) {
        var availableRoles = isc.Auth.getAvailableRoles();
        var valueMap = {
            "_any_": "Any user - no roles required"
        };
        if (availableRoles) {
            availableRoles.sort();
            for (var i = 0; i < availableRoles.length; i++) {
                var role = availableRoles[i];
                valueMap[role] = role;
            }
        }
        valueMap["*super*"] = "SuperUser only";
        valueMap["false"] = "None - no user may access";

        var specialFieldValues = this.specialFieldValues;
        var field = {
            name: fieldName,
            type: "select",
            title: title,
            multiple: true,
            multipleAppearance: "picklist",
            valueMap: valueMap,
            pickListProperties: {
                selectionChanged : function (record, state) {
                    if (state) {
                        var value = record[fieldName];
                        if (specialFieldValues.contains(value)) {
                            // Selecting a special value, clear all other selections, if any
                            var records = this.getSelection();
                            for (var i = 0; i < records.length; i++) {
                                var record = records[i];
                                if (record[fieldName] != value) {
                                    this.deselectRecord(record);
                                }
                            }
                        } else {
                            // Selecting a role, clear any special value selections, if any
                            var records = this.getSelection();
                            for (var i = 0; i < records.length; i++) {
                                var record = records[i];
                                if (specialFieldValues.contains(record[fieldName])) {
                                    this.deselectRecord(record);
                                }
                            }
                        }
                    }
                }
            }
        }
        return field;
    },

    edit : function (field, callback) {
        if (!this.isDrawn()) {
            this.delayCall("edit", arguments);
            return;
        }

        // to be called when editing completes
        this.saveCallback = callback;

        // Convert comma-separated list of roles to a string array for editing
        if (field.viewRequiresRole) {
            field.viewRequiresRole = field.viewRequiresRole.split(",");
        }
        if (field.editRequiresRole) {
            field.editRequiresRole = field.editRequiresRole.split(",");
        }

        // Pull special, no user may access, value from alternate field.
        if (field.viewRequires == "false") {
            field.viewRequiresRole = "false";
            delete field.viewRequires;
        }
        if (field.editRequires == "false") {
            field.editRequiresRole = "false";
            delete field.editRequires;
        }

        // Show default case as if it was selected
        if (field.viewRequiresRole == null) field.viewRequiresRole = "_any_";
        if (field.editRequiresRole == null) field.editRequiresRole = "_any_";

        this.form.editRecord(field);
    },

    cancelClick : function () {
        // This editor is typically embedded in a Window that has a body layout.
        // Find the Window object, if any, and close it.
        var parents = this.getParentElements();
        if (parents && parents.length > 0) {
            var window = parents[parents.length-1];
            if (isc.isA.Window(window)) window.closeClick();
        }
    },

    saveClick : function () {
        if (!this.form.validate()) return;

        var field = this.form.getValues();

        // Process special values
        var value = field.viewRequiresRole;
        if (isc.isAn.Array(value) && value.length == 1) value = value[0];

        delete field.viewRequires;
        if (value == "_any_") {
            delete field.viewRequiresRole;
        } else if (value == "false") {
            field.viewRequires = "false";
            delete field.viewRequiresRole;
        }

        var value = field.editRequiresRole;
        if (isc.isAn.Array(value) && value.length == 1) value = value[0];

        delete field.editRequires;
        if (value == "_any_") {
            delete field.editRequiresRole;
        } else if (value == "false") {
            field.editRequires = "false";
            delete field.editRequiresRole;
        }

        // Convert array of selections to a comma-separated string
        if (isc.isAn.Array(field.viewRequiresRole)) {
            field.viewRequiresRole = field.viewRequiresRole.join(",");
        }
        if (isc.isAn.Array(field.editRequiresRole)) {
            field.editRequiresRole = field.editRequiresRole.join(",");
        }

        this.fireCallback(this.saveCallback, "field", [field]);
        this.saveCallback = null;
    }
},

editFieldSecurity : function (field) {
    if (!this.securityEditor) {
        var securityWindowProperties = {
            title: "Edit security settings for field " + field.name
        };
        this.securityWindow = this.createAutoChild("securityWindow", securityWindowProperties);
        this.securityEditor = this.createAutoChild("securityEditor");
        this.securityWindow.addItem(this.securityEditor);
    }

    var self = this;
    var fieldCopy = isc.shallowClone(field);
    this.securityEditor.edit(field, function (editedField) {
        self.securityWindow.hide();
        // Update field in list by applying only the possibly changed properties being careful
        // to remove the property when it no longer has a value
        if (editedField.viewRequires) field.viewRequires = editedField.viewRequires;
        else if (field.viewRequires) delete field.viewRequires;
        if (editedField.viewRequiresRole) field.viewRequiresRole = editedField.viewRequiresRole;
        else if (field.viewRequiresRole) delete field.viewRequiresRole;

        if (editedField.editRequires) field.editRequires = editedField.editRequires;
        else if (field.editRequires) delete field.editRequires;
        if (editedField.editRequiresRole) field.editRequiresRole = editedField.editRequiresRole;
        else if (field.editRequiresRole) delete field.editRequiresRole;

        // If security really changed, update the sample data grid
        if (fieldCopy.viewRequires != field.viewRequires ||
            fieldCopy.viewRequiresRole != field.viewRequiresRole ||
            fieldCopy.editRequires != field.editRequires ||
            fieldCopy.editRequiresRole != field.editRequiresRole)
        {
            self.saveFieldChanges(["viewRequires","viewRequiresRole","editRequires","editRequiresRole"], fieldCopy, field);
            self.rebindSampleData();
        }
    });

    this.securityWindow.show();
},

relationEditorDefaults: {
    _constructor: "RelationEditor",
    autoDraw: false,
    editComplete : function (canceled) {
        var dsEditor = this.dsEditor,
            fieldEditor = dsEditor.fieldEditor
        ;

        if (!canceled) {
            if (this.readOnly) {
                isc.warn("Project is read-only so DataSource changes are being ignored.");
                return;
            }

            dsEditor.relationEditorWindow.hide();

        }
        // Close window
        dsEditor.relationEditorWindow.closeClick();

        // At this point, the RelationEditor has pushed all changes into the session
        // which is shared with the DataSourceEditor. This includes any field changes
        // on the current DataSource or any other.

        // If there were changes made to the current DataSource, the fields grid needs
        // to be rebound
        var defaults = this.session.getDataSourceDefaults(this.dataSource.ID),
            fields = defaults.fields
        ;

        // Re-bind defaults to FieldEditor
        dsEditor.bindFields(defaults);

        // Re-create testDS with updated fields/data
        if (dsEditor.editSampleData) {
            dsEditor._rebindSampleDataGrid(fields, defaults.cacheData);
        }

        // Update selections and possibly value of titleField. Must come after sampleData updated
        dsEditor.updateTitleField();
        dsEditor.fieldEditor.updateButtonStates();
        dsEditor.fieldEditor.updateIncludeFieldButtonState();

        // Save editor changes (including the updated fields/sample data) to session
        dsEditor.pushEditsToSession();

        // Show notifications for any new relations at lower-left section of fields list
        var fieldsList = fieldEditor.grid,
            x = fieldsList.body.getPageLeft(),
            y = fieldsList.body.getPageTop() + (fieldsList.body.getVisibleHeight()*.66)
        ;
        this.showNewRelationNotifications(x, y, function (dsId) {
            // User clicked to view DS details.
            dsEditor.openDataSource(dsId);
        }, true);
    },

    navigateToDataSource : function (dsName, liveDS, defaults) {
        // Close window
        this.dsEditor.relationEditorWindow.closeClick();

        // Navigate to the target DataSource
        this.dsEditor.navigateToDataSource(dsName, liveDS, defaults);
    }
},

relationEditorWindowDefaults: {
    _constructor: isc.Window,
    autoDraw: false,

    isModal: true, showModalMask: true,
    showHeaderIcon: false,
    showMinimizeButton: false,
    keepInParentRect: true,
    autoCenter:true,
    width: 875,
    height: 760,
    bodyProperties: {
        layoutLeftMargin: 5,
        layoutRightMargin: 5,
        layoutBottomMargin: 5
    },
    canDragResize:true
},

editRelations : function (selectForeignKey) {
    var dsEditor = this;

    this.validateForSave(function (defaults) {
        // Save editor changes to session
        dsEditor.pushEditsToSession(defaults);

        if (!dsEditor.relationEditor) {
            var relationEditorWindowProperties = {
                title: "Relations for DataSource '" + defaults.ID + "'"
            }
            dsEditor.relationEditorWindow = dsEditor.createAutoChild("relationEditorWindow", relationEditorWindowProperties);
            dsEditor.relationEditor = dsEditor.createAutoChild("relationEditor", {
                dsEditor: dsEditor,
                dsDataSource: dsEditor.dsDataSource,
                ownerId: dsEditor.ownerId,
                session: dsEditor.session,
                readOnly: dsEditor.readOnly
            });
            dsEditor.relationEditorWindow.addItem(dsEditor.relationEditor);
        } else {
            dsEditor.relationEditorWindow.setTitle("Relations for DataSource '" + defaults.ID + "'");
        }

        var ds = dsEditor.getCurrentDataSource();
        dsEditor.relationEditor.edit(ds, selectForeignKey);

        dsEditor.relationEditorWindow.show();
    });
},

// Some helper methods to handle fields where the field name may be from field.name or
// implied by an includeFrom value
getFieldNames : function (fields) {
    var _this = this;
    return fields.map(function (field) {
        return _this.getFieldName(field);
    });
},

getFieldName : function (field) {
    if (field.name) return field.name;
    var includeFrom = field.includeFrom,
        parts = includeFrom && includeFrom.split(".")
    ;
    return parts && parts[parts.length-1];
},

findField : function (fields, fieldName) {
    for (var i = 0; i < fields.length; i++) {
        var field = fields[i],
            name = this.getFieldName(field)
        ;
        if (name == fieldName) {
            return field;
        }
    }
},

includeFieldEditorDefaults: {
    _constructor: "VLayout",
    autoDraw: false,
    width: 600,

    formDefaults: {
        _constructor: "DynamicForm",
        autoDraw: false,
        height: 150,
        wrapItemTitles: false,
        numCols: 4,
        colWidths: [ 125, 25, 50, "*" ],
        fields: [
            { name: "relatedDataSource", type: "text", editorType: "SelectItem", title: "Related DataSource",
                colSpan: 3, width: 300, sortField: 0, required: true
            },
            { name: "relatedField", type: "text", editorType: "SelectItem", title: "Field",
                colSpan: 3, width: 300, required: true,
                readOnlyWhen: {
                    _constructor: "AdvancedCriteria", operator: "or",
                    criteria: [
                        { fieldName: "relatedDataSource", operator: "isNull" }
                    ]
                }
            },
            { name: "enableNameAs", type: "boolean", align: "right", width: 125,
                showTitle: false, labelAsTitle: true, startRow: true
            },
            { name: "includeField", type: "text", title: "Name as",  colSpan: 2,
                hint: "in DataSource dsID",
                readOnlyWhen: {
                    _constructor: "AdvancedCriteria", operator: "or",
                    criteria: [
                        { fieldName: "relatedField", operator: "isNull" },
                        { fieldName: "enableNameAs", operator: "notEqual", value: true }
                    ]
                },
                validators : [
                    { 
                        type: "regexp",
                        expression: "^[a-zA-Z_][a-zA-Z0-9_]*$",
                        errorMessage: "Field name must be a valid JavaScript identifier"
                    }
                ]
            },
            { name: "fieldTitle", type: "text", title: "Title as",  colSpan: 3,
                readOnlyWhen: {
                    _constructor: "AdvancedCriteria", operator: "or",
                    criteria: [
                        { fieldName: "relatedField", operator: "isNull" }
                    ]
                }
            },
            { name: "includeSummaryFunction", type: "text", editorType: "SummaryFunctionItem",
                title: "Combine record values as", 
                colSpan: 3, width: 400,
                showIf : function (item, value, form, values) {
                    var relatedDSItem = form.getField("relatedDataSource"),
                        displayValue = relatedDSItem.getDisplayValue(relatedDSItem.getValue()),
                        session = form.creator.creator.session,
                        relatedDS = session.get(relatedDSItem.getValue())
                    ;
                    return (relatedDS && relatedDS.canAggregate != false &&
                            relatedDS.allowClientRequestedSummaries != false &&
                            displayValue && displayValue.contains("(1-to-many)"));
                },
                readOnlyWhen: {
                    _constructor: "AdvancedCriteria", operator: "and",
                    criteria: [
                        { fieldName: "relatedField", operator: "isNull" }
                    ]
                }
            }
        ],
        editNewRecord : function (record) {
            this.updateRelatedFieldChoices(record);
            this.updateNameAsValue(record);
            this.updateNameAsHint(record);
            this.Super("editNewRecord", arguments);
        },
        itemChanged : function (item, newValue) {
            var record = this.getValues();
            if ("relatedDataSource" == item.name) {
                this.updateRelatedFieldChoices(record, true);
                this.updateNameAsValue(record);
                this.updateNameAsHint(record);
                this.clearSummaryFunctionValue();
                this.markForRedraw();   // Re-evaluate includeSummaryFunction showIf()
            } else if ("relatedField" == item.name) {
                this.updateNameAsValue(record);
                this.updateFieldTitleValue();
                this.clearSummaryFunctionValue();
            }
        },
        updateRelatedFieldChoices : function (record, dsChanged) {
            if (!record) return;
            var dsId = record.relatedDataSource;
            if (!dsId) return;
            var session = this.creator.creator.session,
                ds = session.get(dsId)
            ;
            if (ds) {
                var fieldNames = ds.getFieldNames();
                this.getField("relatedField").setValueMap(fieldNames);
            }
            this.clearValue("relatedField");
            this.clearValue("enableNameAs");
            this.clearValue("includeField");
            delete record.relatedField;
            delete record.includeField;
        },
        updateNameAsValue : function (record) {
            if (!record) return;
            var relatedDSId = record.relatedDataSource,
                set = false
            ;
            if (relatedDSId) {
                // No need for a default NameAs value unless the relatedField conflicts
                // with a local field.
                var ds = this.creator.dataSource,
                    relatedFieldValue = record.relatedField
                ;
                if (!relatedFieldValue) return;

                var getExistingField = function (name) {
                    var fields = ds.fields;
                    if (isc.isAn.Object(fields)) {
                        var fieldsArray = [];
                        for (var key in fields) {
                            fieldsArray.add(fields[key]);
                        }
                        fields = fieldsArray;
                    }
                    // Check for an exact case match
                    var localField = fields.find("name", name);
                    if (!localField) {
                        // Perform case-insensitive match against names.
                        // An includeFrom field doesn't need an explicit name.
                        name = name.toLowerCase();
                        for (var i = 0; i < fields.length; i++) {
                            var item = fields.get(i), 
                                itemName = item.name;
                            if (!itemName && item.includeFrom) {
                                itemName = item.includeFrom;
                                var dotIndex = itemName.lastIndexOf(".");
                                if (dotIndex >= 0) itemName = itemName.substring(dotIndex + 1);
                            }
                            if (itemName && itemName.toLowerCase() == name) {
                                localField = item;
                                break;
                            }
                        }
                    }
                    return localField;
                };

                // Existing field using the same name. Introduce an alias based on the
                // target DS and field name
                var relatedDSPrefix = relatedDSId.substring(0, 1).toLowerCase() +
                    relatedDSId.substring(1);
                var includeField = (relatedFieldValue.toLowerCase().startsWith(relatedDSId.toLowerCase()) ?
                    relatedFieldValue :
                    relatedDSPrefix + relatedFieldValue.substring(0, 1).toUpperCase() +
                        relatedFieldValue.substring(1));

                var baseIncludeField = includeField,
                    count = 2;
                while (getExistingField(includeField)) {
                    includeField = baseIncludeField + count++;
                }

                this.setValue("includeField", includeField);
                this.setValue("enableNameAs", true);
                set = true;
            }
            if (!set) {
                this.clearValue("includeField");
                this.clearValue("enableNameAs");
            }
        },
        updateNameAsHint : function (record) {
            if (!record) return;
            var relatedDSId = record.relatedDataSource,
                hint
            ;
            if (relatedDSId) {
                hint = (relatedDSId ? "in <i>" + this.creator.dsId + "</i>" : null);
            }
            this.getField("includeField").setHint(hint);
        },
        updateFieldTitleValue : function () {
            var session = this.creator.creator.session,
                relatedDSId = this.getValue("relatedDataSource"),
                relatedDS = session.get(relatedDSId),
                relatedTitle = (relatedDS ? relatedDS.title || relatedDS.ID.replace(/\d+$/, "") : null),
                relatedFieldValue = this.getValue("relatedField"),
                relatedField = relatedDS.getField(relatedFieldValue),
                title = relatedTitle
            ;
            title = relatedField && (relatedField.title || isc.DS.getAutoTitle(relatedField.name));
            this.setValue("_relatedFieldTitle", title);
            if (!title || !title.toLowerCase().contains(relatedTitle.toLowerCase())) {
                title = relatedTitle + " " + title;
            }

            this.setValue("fieldTitle", title);
            this.setValue("_generatedFieldTitle", title);
        },
        clearSummaryFunctionValue : function () {
            this.clearValue("includeSummaryFunction");

            var session = this.creator.creator.session,
                relatedDSId = this.getValue("relatedDataSource"),
                relatedDS = session.get(relatedDSId),
                relatedFieldValue = this.getValue("relatedField"),
                relatedField = relatedDS.getField(relatedFieldValue),
                type = (relatedField ? relatedField.type : null) || "text"
            ;
            this.getField("includeSummaryFunction").setFieldType(type);
        }
    },

    buttonLayoutDefaults: {
        _constructor: "HLayout",
        autoDraw: false,
        width: "100%",
        height:42,
        layoutMargin:10,
        membersMargin:10,
        align: "right"
    },

    cancelButtonDefaults: {
        _constructor: "IButton",
        autoDraw: false,
        title: "Cancel",
        width: 75,
        autoParent: "buttonLayout",
        click: function() {
            this.creator.cancelClick();
        }
    },

    saveButtonDefaults: {
        _constructor: "IButton",
        autoDraw: false,
        title: "Save",
        width: 75,
        autoParent: "buttonLayout",
        click: function() {
            this.creator.saveClick();
        }
    },

    initWidget : function () {
        this.Super('initWidget', arguments);

        this.addAutoChildren(["form","buttonLayout"]);
        this.buttonLayout.addMember(this.createAutoChild("cancelButton"));
        this.buttonLayout.addMember(this.createAutoChild("saveButton"));
    },

    edit : function (ds, relations, callback, selectDataSource) {
        if (!this.isDrawn()) {
            this.delayCall("edit", arguments);
            return;
        }
        var relationTypeMap = {
            "1-M": "1-to-many",
            "M-1": "many-to-1",
            "Self": "tree self-relation"
        };

        // provide an up-to-date valueMap of related datasources for selection
        var dsId = ds.ID,
            valueMap = {}
        ;
        relations.map(function (r) {
            valueMap[r.dsId] = r.dsId + " (" + relationTypeMap[r.type] + ")";
        });
        this.form.getField("relatedDataSource").setValueMap(valueMap);
        if (!selectDataSource) {
            var choices = isc.getKeys(valueMap);
            if (choices && choices.length == 1) selectDataSource = choices[0];
        } 

        // Hang on to the dsId and relations so type can be looked up upon selection
        this.dataSource = ds;
        this.dsId = dsId;
        this.relations = relations;

        // to be called when editing completes
        this.saveCallback = callback;

        this.form.editNewRecord({ relatedDataSource: selectDataSource });
    },

    cancelClick : function () {
        // This editor is typically embedded in a Window that has a body layout.
        // Find the Window object, if any, and close it.
        var parents = this.getParentElements();
        if (parents && parents.length > 0) {
            var window = parents[parents.length-1];
            if (isc.isA.Window(window)) window.closeClick();
        }
    },

    saveClick : function () {
        if (!this.form.validate()) return;

        var values = this.form.getValues();

        // There could be multiple relation paths to reach the target related data source.
        // Although there is likely not much difference between the paths an effort is made
        // to find the shortest path using a relationship tree.
        var source = this.dataSource;
        var target = values.relatedDataSource;
        var shortest = source.getDefaultPathToRelation(target, this.relations);
        var relationPath = shortest.path.replaceAll('/', '.');
        // Using the calculated relationPath and field name, create a new DS field for the
        // includeFrom
        var field = {
            name: values.includeField || values.relatedField,
            includeFrom: relationPath + "." + values.relatedField
        };
        if (values.fieldTitle && values.fieldTitle != values._relatedFieldTitle) {
            field.title = values.fieldTitle;
        }
        if (values.includeSummaryFunction) {
            field.includeSummaryFunction = values.includeSummaryFunction;
        }
        this.fireCallback(this.saveCallback, "field", [field]);
        this.saveCallback = null;
    }
},

includeFieldEditorWindowDefaults: {
    _constructor: isc.Window,
    autoDraw: false,
    title: "Add included field",

    isModal: true, showModalMask: true,
    showHeaderIcon: false,
    showMinimizeButton: false,
    keepInParentRect: true,
    autoCenter:true,
    autoSize:true,
    canDragResize:true,
    closeClick : function () {
        if (this.dropDS) {
            this.dropDS.destroy();
        }
        this.close();
    }
},

editIncludeField : function (selectDataSource) {
    if (!this.includeFieldEditor) {
        this.includeFieldEditorWindow = this.createAutoChild("includeFieldEditorWindow");
        this.includeFieldEditor = this.createAutoChild("includeFieldEditor", {
            dsDataSource: this.dsDataSource,
            ownerId: this.ownerId
        });
        this.includeFieldEditorWindow.addItem(this.includeFieldEditor);
    }

    var dsId = this.mainEditor.getValue("ID"),
        dsRelations = this.session.getRelations(),
        relations = dsRelations.getAllRelationsForDataSource(dsId),
        // relations = this.dsRelations.getAllRelationsForDataSource(dsId),
        relationTypeMap = isc.DSRelations.relationTypeDescriptionMap
    ;

    // provide an up-to-date valueMap of related datasources for selection
    var valueMap = {};
    relations.map(function (r) {
        valueMap[dsId] = r.dsId + " (" + relationTypeMap[r.type] + ")";
    });

    var ds = this.session.get(dsId);

    var self = this;
    this.includeFieldEditor.edit(ds, relations, function (field) {
        self.includeFieldEditorWindow.hide();
        // Add new field to list
        self.fieldEditor.newRecord(field);

        if (self.includeFieldEditorWindow.dropDS) {
            self.includeFieldEditorWindow.dropDS.destroy();
        }
    }, selectDataSource);

    this.includeFieldEditorWindow.show();
},

formatEditorDefaults: {
    _constructor: "NumberFormatEditor",  // May change at invocation time
    autoDraw: false
},

formatWindowDefaults: {
    _constructor: isc.Window,
    autoDraw: false,

    isModal: true, showModalMask: true,
    showHeaderIcon: false,
    showMinimizeButton: false,
    keepInParentRect: true,
    autoCenter:true,
    autoSize:true,
    canDragResize:true
},

editFormatting : function (field) {
    var type = this.getFieldType(field),
        actualDataType = field.type == "datetime" ? field.type : type,
        isDate = type == "date" || type == "time",
        create;
    if (this.formatEditor == null) {
        create = true;
    } else if (isDate && isc.isA.NumberFormatEditor(this.formatEditor) ||
                !isDate && isc.isA.DatetimeFormatEditor(this.formatEditor)) 
    {
        this.formatEditor.destroy();
        create = true;
    } else if (isDate && actualDataType != this.formatEditor.getActualDataType()) {
        this.formatEditor.destroy();
        create = true;
    }
    if (create) {
        var formatEditorProps = {
            _constructor: type == "date" || type == "time" ? "DatetimeFormatEditor" : "NumberFormatEditor"
        }
        this.formatWindow = this.createAutoChild("formatWindow");
        this.formatEditor = this.createAutoChild("formatEditor", formatEditorProps);
        this.formatWindow.addItem(this.formatEditor);
    }

    this.formatWindow.setTitle("Edit formatting for " + field.name);

    var self = this;
    this.formatEditor.edit(field, actualDataType, function (editedField) {
        self.formatWindow.hide();
        var oldField = self.fieldEditor.grid.data.getCleanNodeData(field);
        // Update field in list by applying only the possibly changed properties being careful
        // to remove the property when it no longer has a value
        if (editedField.format) field.format = editedField.format;
        else if (field.format) delete field.format;
        self.saveFieldChanges("format", oldField, field);
        self.fieldEditor.updateButtonStates(field);
        self.rebindSampleData();
    }, function() {
        self.formatWindow.hide();
    });

    this.formatWindow.show();
},

// move a field up or down by <delta> rows
moveField : function (node, delta) {
    var grid = this.fieldEditor.grid,
        tree = grid.data,
        parentNode = tree.getParent(node),
        currentPosition = tree.findNodeIndex(node),
        totalNodes = tree.getChildren(parentNode).length,
        newPosition = Math.min(Math.max(0, currentPosition + delta), totalNodes)
    ;
    if (newPosition != currentPosition) {
        var nodeRow = grid.getRecordIndex(node),
            editRow = grid.getEditRow()
        ;
        if (editRow != null && editRow == nodeRow) {
            var message = "The field to be moved is has pending edits.<br>Save them now?";
            isc.warn(message, function (value) {
                if (value) {
                    grid.endEditing();
                    tree.move(node, parentNode, newPosition);
                }
            }, {
                buttons: [
                    isc.Dialog.CANCEL,
                    isc.Dialog.OK
                ]
            });
    
        } else {
            tree.move(node, parentNode, newPosition);
        }
    }
},

saveFieldChanges : function (properties, oldField, newField) {
    properties = (isc.isAn.Array(properties) ? properties : [properties]);

    var session = this.session,
        dsName = this.getDataSourceID(),
        fieldName = oldField.name,
        changeContext;
    for (var i = 0; i < properties.length; i++) {
        var property = properties[i],
            oldValue = oldField[property],
            newValue = newField[property],
            context = null
        ;
        if (oldValue != newValue) {
            if (property == "name") {
                context = session.addChange(dsName, "changeFieldName", {
                    fieldName: oldValue,
                    newName: newValue
                });
            } else if (property == "type" || property == "length") {
                // Not handled here
            } else {
                context = session.addChange(dsName, "changeFieldProperty", {
                    fieldName: fieldName,
                    property: property,
                    originalValue: oldValue,
                    newValue: newValue
                });
            }
            if (!changeContext && context) {
                changeContext = context;
                session.startChangeContext(changeContext);
            }
        }
    }
    if (changeContext) session.endChangeContext(changeContext);
},

// ---------------------------------------------------------------------------------------
// 

createLiveDSInstance : function (dsData) {
    var dsClass = this.dsClass || dsData._constructor || "DataSource",
        schema;
    if (isc.DS.isRegistered(dsClass)) {
        schema = isc.DS.get(dsClass);
    } else {
        schema = isc.DS.get("DataSource");
        dsData._constructor = dsClass;
    }

    // create a live instance
    var liveDS = isc.ClassFactory.getClass(dsClass).create(dsData);
    
    return liveDS;
},

//> @method dataSourceEditor.cancelClick
// Method called when the cancel button is clicked (assuming +link{showActionButtons} is
// enabled). By default +link{editComplete} is called with the <code>canceled</code>
// argument set to <code>true</code>.
//
// @visibility reifyOnSite
//<
cancelClick : function () {
    if (this.editComplete) {
        // If this editor instance created the session, clear changes because they aren't saved
        if (this._createdSession) {
            this.session.clearChangeLog();
        }
        this.editComplete(true);
    }
},

//> @method dataSourceEditor.saveClick
// Method called when the save button is clicked (assuming +link{showActionButtons} is
// enabled). By default the editors are +link{validate,validated}, the resulting
// DataSource is +link{validateForSave,validated to be saved} and finally +link{editComplete}
// is called with the <code>canceled</code> argument set to <code>true</code> if no changes
// are pending for the current DataSource or any others that have been edited from this
// session. Otherwise editComplete() is called to allow the caller to determine the next
// steps.
// <P>
// It is the responsibility of the caller to call +link{save} to persist changes if desired.
// See +link{editComplete} for a description of the save process.
// <P>
// Note that if the +link{session} is being shared with another component, the changes are
// reflected there and can be saved by any of the components.
//
// @visibility reifyOnSite
//<
saveClick : function () {
    if (!this.validate()) return false;

    var dsEditor = this,
        session = this.session
    ;
    if (this.editSampleData) {
        // Make sure any active editors are saved
        dsEditor.dataGrid.endEditing();
    }

    this.validateForSave(function (defaults) {
        // Save editor changes to session
        dsEditor.pushEditsToSession(defaults);

        // Instead of saving, report a cancel if nothing has changed
        var canceled = (session.getAddedDataSources().length == 0 &&
                        session.getUpdatedDataSources().length == 0 &&
                        isc.isAn.emptyObject(session.getRenamedDataSources()));
        if (dsEditor.editComplete) {
            dsEditor.editComplete(canceled);
        }
    });
},

//> @method dataSourceEditor.validate
// Validates that the main DataSource and fields editors. Additional validation
// for saving can be performed by calling +link{validateForSave}.
//
// @visibility reifyOnSite
//<
validate : function () {
    var valid=true;
    if (this.showMainEditor != false) {
        valid = this.mainEditor.validate();
    } else if (this.showMockEditor != false) {
        valid = this.mockEditor.validate();
    }
    var fieldEditor = this.fieldEditor;
    return (fieldEditor.validate() && valid);
},

//> @method dataSourceEditor.validateForSave
// Performs a number of validation and user confirmations so that the current DataSource
// can be saved.
// <ol>
//   <li>Validates the current editors via +link{validate}.</li>
//   <li>If the DataSource ID has changed and the new ID conflicts with an existing
//       DataSource, confirms with the user that it can be overwritten.</li>
//   <li>If +link{makeUniqueTableName} is enabled and this editor is not in +link{readOnly}
//       mode, the +link{dataSource.tableName} is made unique is needed.</li>
//   <li>If the DataSource ID has changed and +link{showWarningOnRename} is enabled,
//       the user is shown a +link{renameWarningMessage,warning message} and required
//       to confirm for saving.</li>
// </ol>
//
// @param callback (Callback) Callback to fire only if valid for saving. Takes a single
//                            parameter <code>defaults</code> - the defaults that can be saved.
// @visibility reifyOnSite
//<
validateForSave : function (callback) {
    var dsEditor = this,
        defaults = this.getDefaults()
    ;
    if (this.validate()) {
        dsEditor.verifyOverwrite(defaults, function () {
            dsEditor.verifyUniqueTableName(defaults, function () {
                dsEditor.warnOnRename(defaults, function () {
                    if (callback != null) callback(defaults);
                });
            });
        });
    }
},

//> @method dataSourceEditor.pushEditsToSession
// This method will push all unsaved edits into the +link{dataSourceEditor.session}.
//
// @visibility reifyOnSite
//<

pushEditsToSession : function (defaults) {
    // Get full defaults including sample data
    defaults = defaults || this.getDefaults();
    if (!defaults) {
        return false;
    }
    this._pushingEditsToSession = true;

    var previousID = this._currentDSName;

    var isRenamed = (defaults.ID != previousID),
        savedToSession = this.session.getDataSourceDefaults(defaults.ID)
    ;
    if (isRenamed) {
        // Update the session with the new/renamed defaults
        this.session.rename(previousID, defaults.ID, defaults, this._editingNewDataSource);
        this._currentDSName = defaults.ID;

    } else if (!savedToSession) {
        // Update the session with the new defaults
        this.session.add(defaults.ID, defaults, !this._editingNewDataSource);
    } else {
        // Update the session with the defaults
        this.session.set(defaults.ID, defaults);
    }
    delete this._pushingEditsToSession;

    return defaults;
},

//> @method dataSourceEditor.save
// Saves all pending DataSource changes to +link{dsDataSource}. The +link{dataSourceSaved}
// method is called after each DataSource is saved.
// <P>
// The callback will be called even if there are no pending changes to save as long as the
// current editors are +link{validate,valid}.
//
// @param [callback] (Callback) the callback function to be called when all saves complete
// @return (Boolean) false if the current editors are not valid. No callback will be made
//                   in this case either.
// @visibility reifyOnSite
//<
save : function (callback) {
    // Wait until edit is completely initialized
    if (!this._ready) {
        this.delayCall("save", [callback], 250);
        return;
    }

    if (this.editSampleData) {
        // Make sure any active editors are saved
        this.dataGrid.endEditing();
    }

    if (!this.validate()) {
        return false;
    }

    // Save editor changes to session
    this.pushEditsToSession();

    // Save session changed to dsDataSource
    this.session.saveChanges(callback, { target: this, methodName:"dataSourceSaved" });
},

//> @method dataSourceEditor.acceptChanges
// Marks all changes as accepted so that the +link{session} has no pending changes. Nothing is saved
// or updated - only pending state is cleared (see +link{DataSourceEditorSession.acceptChanges}).
//
// @visibility reifyOnSite
//<
acceptChanges : function () {
    this.session.acceptChanges();
},

//> @method dataSourceEditor.rollbackChanges
// Revert session changes to their original loaded state. This includes reloading changed DataSource
// definitions from storage. Doing so will cause the session to transition to not
// +link{DataSourceEditorSession.isReady()} state while reloading and any
// +link{DataSourceEditorSession.waitForReady} callbacks will be fired once the reloads are
// complete. Immediately thereafter the method callback, if provided, is called.
// <P>
// If only the edit state of the session should be cleared maintaining the current edits, see
// +link{acceptChanges}.
//
// @param callback (Callback) callback to fire on completion
// @visibility reifyOnSite
//<
rollbackChanges : function (callback) {
    this.session.rollbackChanges(callback);
},

getFieldRenames : function (name) {
    return this.session.getRenamedFields(name);
},

getFieldAdds : function (name) {
    return this.session.getAddedFields(name);
},

getFieldDeletes : function (name) {
    return this.session.getRemovedFields(name);
},

getCurrentDataSourceID : function () {
    return this.mainEditor.getValue("ID");
},

getCurrentDataSource : function () {
    return this.session.get(this.getCurrentDataSourceID());
},

//> @method dataSourceEditor.getDefaults
// Return default properties for a dataSource DataSource.
// By default this returns the dataSource currently being edited. Pass in an
// explicit name from the +link{knownDataSources} to pick up the defaults for
// another known dataSource in the session.
// <P>
// For the currently edited DataSource, null is returned 
// if a primary key is required but has not been configured.
//
// @param [name] (Identifier) the DataSource ID to query. Use null to get currently edited
//                          DataSource defaults.
// @return (Properties) the current edited DataSource defaults
// @visibility reifyOnSite
//<
getDefaults : function (name) {
    if (name) {
        return this.session.getDataSourceDefaults(name);
    }

    var fieldEditor = this.fieldEditor;
    if (fieldEditor.isVisible()) {
        fieldEditor.saveRecord();
    }

    var defaults = this.getDatasourceData();

    // When field editor is visible (i.e. not a basic MockDataSource)
    // validate that there is a PK or add one
    if (fieldEditor.isVisible()) {
        // Determine if there is a defined field marked as PK or
        // DataSource inherits a PK field.
        var allFields = this.getAllFields(defaults),
            hasPK = this.hasPrimaryKeyField(allFields)
        ;
        if (!hasPK) {
            if (this.autoAddPK) {
                this.createUniqueIdField(defaults.fields);

            } else if (this.requiresPrimaryKey(defaults)) {

                isc.warn("DataSource must have a field marked as the primary key");
                return null;
            }
        }
    }

    // Possibly hacky fix for a problem saving these values when they are null ...
    ["recordXPath", "dataURL", "dbName", "schema", "tableName", "quoteTableName", "beanClassName", "dropExtraFields", "autoDeriveSchema"].map(function (removeNull) {
        if (defaults[removeNull] == null) delete defaults[removeNull];
    });

    // And remove _constructor: DatabaseBrowser if present ... not sure where that comes from
    if (defaults._constructor == "DatabaseBrowser") delete defaults._constructor;

    // Sample data for client-only DataSources has includeFrom values resolved locally
    // for existing values or new records. The includeFrom values should not be saved
    // as part of the DataSource. Remove these values here even if editing of sample
    // data is not enabled.
    if (defaults.cacheData) {
        var fields = defaults.fields;
        for (var i = 0; i < fields.length; i++) {
            var field = fields[i];
            if (field.includeFrom) {
                defaults.cacheData.map(function(currentValue, index, arr) {
                    delete currentValue[field.name]
                });
            }
        }
    }

    // Update cacheData based on security properties if we are editing SampleData.
    if (this.editSampleData) {
        // This is done by processing each data record through clientOnly response handling
        // which requires a live DS with the properties we have only in defaults so far.
        // Create a temp DataSource for this processing.
        var ds = this.session.createTempLiveDSInstance(defaults),
           targetCacheData = defaults.cacheData || []
        ;

        // no need to calculate includeFrom values
        ds.includeFromValuesProvided = true;

        var session = this.session;
        isc.DS.addClassMethods({
            _getDataSource : isc.DS.getDataSource,
            getDataSource : function (name, callback, context, schemaType) {
                var inGetDataSource = this._inGetDataSource;
                this._inGetDataSource = true;
                var ds = (inGetDataSource ? this._getDataSource(name) : session.get(name));
                delete this._inGetDataSource;
                return ds;
            }
        });

        for (var i = 0; i < targetCacheData.length; i++) {
            var record = targetCacheData[i],
                op = record._operation
            ;
            switch (op) {
                case "remove":
                    ds.getClientOnlyResponse({operationType: 'remove', data: record});
                    break;
                case "update":
                    ds.getClientOnlyResponse({operationType: 'update', data: record});
                    break;
                default:
                    // This handles explicit "adds" and any new record that has no _operation
                    
                    var response = ds.getClientOnlyResponse({operationType: 'add', data: record, clientOnlyDataModified: true});
                    if (response.status == 0) {
                        ds.cacheData[i] = response.data;
                    }
                    break;
            }
        }

        // Grab the security-updated data back from the temporary DataSource
        defaults.cacheData = ds.cacheData;

        isc.DS.addClassMethods({
            getDataSource : isc.DS._getDataSource,
            _getDataSource : null
        });

        // The temp DS is no longer needed
        ds.destroy();
    }

    return defaults;
},

//> @method dataSourceEditor.getDataSourceXml
// Return DataSource definition as XML.
// By default, this returns the dataSource currently being edited. Pass in an
// explicit name from the +link{knownDataSources} to pick up the XML for
// another known dataSource in the session.
// <P>
// For the currently edited DataSource, null is returned
// if a primary key is required but has not been configured.
//
// @param [name] (Identifier) the DataSource ID to query. Use null to get currently edited
//                          DataSource XML.
// @return (String) the current edited DataSource XML
// @visibility reifyOnSite
//<
getDataSourceXml : function (name) {
    if (name) {
        return this.session.getDataSourceXml(name);
    }

    const defaults = this.getDefaults();
    return this.session.serializeDataSource(defaults);
},

//> @method dataSourceEditor.requiresPrimaryKey()
// Does this dataSource require a primary key field? Default implementation returns +link{requirePK}
// @visibility reifyOnSite
//<
requiresPrimaryKey : function (defaults) {
    return this.requirePK;
},

// returns true if no check was necessary
verifyOverwrite : function (defaults, callback) {
    // If editing an existing DS and the ID hasn't changed, don't check for uniqueness.
    // Don't check in readOnly mode either.
    if ((this._origDSName != null && this._origDSName == defaults.ID) || this.readOnly) {
        if (callback) callback();
        return true;
    }
    
    // Confirm that the DS ID is unique
    var dataSourceName = defaults.ID;
    this.session.hasDataSourceXml(dataSourceName, function (found) {
        if (!found) {
            // Filename wasn't found so it is unique
            if (callback) callback();
            return;
        }
        // Warn user that continuing will overwrite existing DS
        isc.warn("DataSource name '" + dataSourceName + "' is already in use. " +
                    "Overwrite the existing DataSource?",
        function (value) {
            if (value && callback) callback();
        }, {
            buttons: [
                isc.Dialog.CANCEL,
                { title: "Overwrite", width:75, overflow: "visible",
                    click: function () { this.topElement.okClick() }
                }
            ],
            autoFocusButton: 1
        });
    });
},

warnOnRename : function (defaults, callback) {
    // If renaming DataSource, confirm the change
    if (!this._editingNewDataSource && this.showWarningOnRename &&
        this._origDSName != null && this._origDSName != defaults.ID &&
        !this.readOnly)
    {
        isc.warn(this.renameWarningMessage, function (value) {
            if (value) {
                if (callback) callback(defaults);
            }
        }, {
            buttons: [
                isc.Dialog.CANCEL,
                isc.Dialog.OK
            ]
        });
    } else {
        if (callback) callback(defaults);
    }
},

verifyUniqueTableName : function (defaults, callback, nextSuffix) {
    var origTableName = (this._origDSDefaults ? this._origDSDefaults.tableName || this._origDSName : null),
        tableName = (defaults.tableName || defaults.ID) + (nextSuffix != null ? "_" + nextSuffix : ""),
        _this = this
    ;
    if (!this.makeUniqueTableName || this.readOnly) {
        // No need to verify table name
        if (callback) callback(defaults);
        return;
    }
    if (origTableName && origTableName == tableName) {
        // Editing an existing DataSource and the tableName hasn't changed.
        // No need to verify a unique name.
        if (callback) callback(defaults);
        return;
    }

    

    isc.DMI.call({
        appID: "isc_builtin",
        className: "com.isomorphic.tools.AdminConsole",
        methodName: "tableExists",
        arguments: [ tableName, null ],
        callback : function (request, data) {
            if (data) {
                var suffix = nextSuffix || -1;
                _this.verifyUniqueTableName(defaults, callback, suffix + 1);
            } else {
                if (nextSuffix != null) defaults.tableName = tableName;
                if (callback) callback(defaults);
            }
        }
    });
},

//> @attr dataSourceEditor.uniqueIdFieldName (String : "uniqueId" : IR)
// Name for the automatically added uniqueId primary key field if added if
// +link{autoAddPK} is true
// @visibility reifyOnSite
//<
uniqueIdFieldName:"uniqueId",
uniqueIdFieldType:"sequence",

getUniqueIdFieldConfig : function () {
    var field = {
        primaryKey:true,

        hidden:true,
        name:this.uniqueIdFieldName,
        type:this.uniqueIdFieldType
    };
    return field;
},


createUniqueIdField : function (fields) {
    var field = this.getUniqueIdFieldConfig();
    fields.addAt(field, 0);
},

getExtraCleanNodeData : function (nodeList, includeChildren) {
    if (nodeList == null) return null;

    var nodes = [], 
        wasSingular = false;
    if (!isc.isAn.Array(nodeList)) {
        nodeList = [nodeList];
        wasSingular = true;
    }

    for (var i = 0; i < nodeList.length; i++) {
        var treeNode = nodeList[i],
            node = {};
        // copy the properties of the tree node, dropping some further Tree/TreeGrid artifacts
		for (var propName in treeNode) {
            if (propName == "id" || propName == "parentId" || propName == "isFolder") continue;

            node[propName] = treeNode[propName];

            // Clean up the children as well (if there are any)
            if (propName == this.fieldEditor.grid.data.childrenProperty && isc.isAn.Array(node[propName])) {
                node[propName] = this.getExtraCleanNodeData(node[propName]);
            }
        }
        nodes.add(node);
    }
    if (wasSingular) return nodes[0];
    return nodes;
},

clear : function () {
    if (this.mainEditor) this.mainEditor.clearValues();
    else this.mainEditorValues = null;
    this.fieldEditor.setData([]);
},

//> @method dataSourceEditor.hasPendingChanges
// Does this dataSource editor have unsaved changes? To check for changes to a specific
// dataSource, pass in a dataSource ID from the +link{knownDataSources} array. If no
// name parameter was passed this method will return true if any DataSource has been changed.
//
// @param [name] (String) the DataSource ID to check or null to check all DataSources
// @visibility reifyOnSite
//<
hasPendingChanges : function (name) {
    var session = this.session,
        changed = false
    ;
    if (name) {
        changed = (session.getAddedDataSources().contains(name) ||
                    session.getUpdatedDataSources().contains ||
                    session.getRenamedDataSources()[name] != null);
    } else {
        changed = (session.getAddedDataSources().length > 0 ||
                    session.getUpdatedDataSources().length > 0 ||
                    !isc.isAn.emptyObject(session.getRenamedDataSources()));
    }
    return changed;
},

hasPrimaryKeyChanged : function () {
    return (this.fieldEditor.grid._originalPKFieldName != null);
},

openDataSource : function (dsName) {
    var dsEditor = this,
        session = this.session
    ;

    // Don't navigate away from the current DataSource if it is invalid and cannot be saved
    this.validateForSave(function (defaults) {
        // Save any local changes to session
        dsEditor.pushEditsToSession(defaults);

        var liveDS = session.get(dsName),
            defaults = session.getDataSourceDefaults(dsName)
        ;
        if (dsEditor.navigateToDataSource) {
            dsEditor.navigateToDataSource(dsName, liveDS, defaults);
        }
    });
},

//> @method dataSourceEditor.navigateToDataSource
// Method called when the user requests to edit another DataSource, typically from
// clicking a link in a relation shown when +link{canNavigateToDataSource} is enabled.
// <P>
// By default, the target DataSource is shown for editing in the current editor by
// calling +link{editSaved} directly with the passed <code>defaults</code> letting the
// current +link{session} manage any changes. However, by overriding this method,
// the caller can determine how navigation occurs. An alternative is to open a new
// +link{DataSourceEditor} instance.
//
// @param dsName (String) the DataSource ID to be shown
// @param liveDS (DataSource) the live DS to be shown. This may be a temporary instance.
// @param defaults (Properties) the defaults use to create an instance of the target DataSource
//
// @visibility reifyOnSite
//<
navigateToDataSource : function (dsName, liveDS, defaults) {
    // Sharing the same DSE instance - just start a new edit
    this.editSaved(defaults)
},

initWidget : function () {
    this.Super('initWidget', arguments);

    // When editing sample data put main editor into a tab and add another tab for data.
    // The tabset is always created but is hidden unless editing sample data.
    this.addAutoChild("mainTabSet", { visibility: "hidden" });

    // DataSource Fields tab contents
    this.mainStack = this.createAutoChild("mainStack");
    this.instructions = this.createAutoChild("instructions");

    // Sample Data tab contents
    var label = this.createAutoChild("sampleDataLabel");
    this.dataGrid = this.createAutoChild("sampleDataGrid");

    var addNewButton = this.createAutoChild("sampleDataAddRecordButton");
    var discardDataButton = this.createAutoChild("sampleDataDiscardDataButton");
    var buttonLayout = this.createAutoChild("sampleDataButtonLayout", {
        members: [ addNewButton, discardDataButton ]
    });

    this.dataPane = this.createAutoChild("sampleDataPane", {
        members: [ label, this.dataGrid, buttonLayout ] });

    // Create tabs
    this.mainTabSet.addTab({
        name: "fields",
        title: "DataSource Fields"
        // pane set below if editing sample data
    });
    this.mainTabSet.addTab({
        name: "sampleData",
        title: "Sample Data",
        pane: this.dataPane,
        // When sample data tab is selected, make sure the row being edited
        // receives focus
        tabSelected : function (tabSet, tabNum, tabPane, ID, tab, name) {
            var grid = tabPane.getMember(1),
                editRowNum = grid.getEditRow()
            ;
            if (editRowNum >= 0 && grid.getEditRow() != editRowNum) {
                isc.Timer.setTimeout(function () {
                    grid.startEditing(editRowNum);
                });
            }
        }
    });

    // Show correct initial layout based on editing sample data or not.
    if (this.editSampleData) {
        this.mainTabSet.setTabPane("fields", this.mainStack);
        this.mainTabSet.show();
    } else {
        this.addMember(this.mainStack);
    }

    this.addAutoChild("mainEditor", { 
        fields: this.getMainEditorFields(), 
        showMainEditorSectionHeaders:this.showMainEditorSectionHeaders 
    });
    if (!this.canEditDataSourceID) {
        this.mainEditor.getField("ID").setReadOnlyDisplay("static");
        this.mainEditor.getField("ID").setCanEdit(false);
    }
    if (this.showOperationBindingsEditor) this.addAutoChild("opBindingsEditor");
    this.addAutoChild("mockEditor", { allowExplicitPKInSampleData: this.allowExplicitPKInSampleData });
    this.addAutoChild("buttonLayout");
    this.addTestDataButton = this.createAutoChild("addTestDataButton");
    this.editWithFieldsButton = this.createAutoChild("editWithFieldsButton");
    this.buttonLayout.addMember(this.addTestDataButton);
    this.buttonLayout.addMember(this.editWithFieldsButton);
    this.buttonLayout.addMember(isc.LayoutSpacer.create({ width: 20 }));
    this.buttonLayout.addMember(this.createAutoChild("cancelButton"));
    this.buttonLayout.addMember(this.createAutoChild("saveButton"));
    if (!this.showActionButtons) {
        this.buttonLayout.hide();
    }

    if (this.dsDataSource) this.dsDataSource = isc.DataSource.get(this.dsDataSource);
    
    if (this.canAddChildSchema) {
        this.canEditChildSchema = true;
        this.addAutoChild("addChildButton");
    }

    this.legalValuesButton = this.createAutoChild("legalValuesButton", {
        visibility: (this.showLegalValuesButton ? "inherit" : "hidden")
    });
    this.derivedValueButton = this.createAutoChild("derivedValueButton",
        { visibility: (this.showDerivedValueButton ? "inherit" : "hidden")
    });
    this.validatorsButton = this.createAutoChild("validatorsButton");
    this.formattingButton = this.createAutoChild("formattingButton");
    this.securityButton = this.createAutoChild("securityButton");
    if (this.enableRelationEditor) {
        this.relationsButton = this.createAutoChild("relationsButton");
    }
    if (this.showIncludeFromButton) {
        this.includeFieldButton = this.createAutoChild("includeFieldButton");
    }
    this.reorderButtonLayout = this.createAutoChild("reorderButtonLayout");
    this.reorderUpButton = this.createAutoChild("reorderUpButton");
    this.reorderDownButton = this.createAutoChild("reorderDownButton");
    this.reorderButtonLayout.addMembers([this.reorderUpButton, this.reorderDownButton]);

    this.addAutoChild("fieldEditor", {

        fields: this.getFieldEditorGridFields(),
        formFields: this.getFieldEditorFormFields(),
        // NOTE: provided dynamically because there's currently a forward dependency:
        // DataSourceEditor is defined in ISC_DataBinding but ComponentEditor is defined
        // in ISC_Tools
        formConstructor: isc.TComponentEditor || isc.ComponentEditor,
		gridConstructor: this.canEditChildSchema ? isc.TreeGrid : isc.ListGrid,
        showMoreButton: this.showMoreButton,
        newButtonTitle: "New Field",
		newButtonDefaults: this.newButtonDefaults,
        newButtonProperties: this.newButtonProperties,
		moreButtonDefaults: this.moreButtonDefaults,
        moreButtonProperties: this.moreButtonProperties,
        autoAssignNewFieldName: false
    });
    this.moreButton = this.fieldEditor.moreButton;
    this.newButton = this.fieldEditor.newButton;

    this.fieldEditor.gridButtons.addMembers([this.legalValuesButton, this.derivedValueButton, this.validatorsButton, this.formattingButton, this.securityButton]);
    if (this.canAddChildSchema) this.fieldEditor.gridButtons.addMember(this.addChildButton);
    this.fieldEditor.gridButtons.addMembers([isc.LayoutSpacer.create({ height: 15 }), this.relationsButton, this.includeFieldButton]);
    if (this.canAddChildSchema) this.fieldEditor.gridButtons.addMember(this.addChildButton);
    this.fieldEditor.gridButtons.addMembers([isc.LayoutSpacer.create(), this.reorderButtonLayout, isc.LayoutSpacer.create()]);

    var stack = this.mainStack;

    stack.addSections([isc.addProperties(this.instructionsSectionDefaults,
        this.instructionsSectionProperties,
        { items:[this.instructions] }
    )]);

	stack.addSections([isc.addProperties(this.mainSectionDefaults,
        this.mainSectionProperties,
        { items:[this.mainEditor] }
    )]);
    if (this.showMainEditor==false) stack.hideSection(1);

    stack.addSections([isc.addProperties(this.fieldSectionDefaults,
        this.fieldSectionProperties,
        { items:[this.fieldEditor] }
    )]);

    if (this.showOperationBindingsEditor) {
        stack.addSections([isc.addProperties(this.opBindingsSectionDefaults,
            this.opBindingsSectionProperties,
            { items:[this.opBindingsEditor] }
        )]);
    }

    stack.addSections([isc.addProperties(this.mockSectionDefaults,
        this.mockSectionProperties,
        { items:[this.mockEditor] }
    )]);

    var _this = this;
    this.deriveForm = this.createAutoChild("deriveForm", {
        fields: [
            {name: "sql", showTitle: false, formItemType: "AutoFitTextAreaItem",
             width: "*", height: 40, colSpan: "*",
             keyPress:function (item, form, keyName) {
                if (keyName == 'Enter' && isc.EH.ctrlKeyDown()) {
                   if (isc.Browser.isSafari) item.setValue(item.getElementValue());
                   _this.execSQL();
                   if (isc.Browser.isSafari) return false;
                }
            }},
            {type: "button", title: "Execute", startRow: true, click: this.getID()+".execSQL()"}
        ]
    });

    // Create an editor session if not already provided
    if (!this.session) {
        this.session = this.createAutoChild("session", {
            dsDataSource: this.dsDataSource,
            dataSources: this.knownDataSources,
            useLiveDataSources: this.useLiveDataSources,
            useChangeTracking: this.useChangeTracking
        });
        // Make note that we created the session
        this._createdSession = true;
    }
    this.registerChangeObservations();

    /*
    // disabled - would need to add some instructions and error handling before this can be shown
    stack.addSections([isc.addProperties(this.deriveFieldsSectionDefaults,
        this.deriveFieldsSectionProperties,
        { items:[this.deriveForm] }
    )]);

    //this.operationsGrid = this.createAutoChild("operationsGrid");
    //stack.addSection({ID: "operationsSection", title: "Operations", expanded: false, items: [this.operationsGrid]});

    this.previewGrid = this.createAutoChild("previewGrid");
    stack.addSection({ID: "previewSection", title: "Preview", expanded: false, items: [this.previewGrid]});
    */
},

destroy : function () {
    // Make sure sampleData testDS and temporary fields DS are destoyed
    if (this.testDS) this.testDS.destroy();
    if (this._fieldsDS) this._fieldsDS.destroy();
    // Drop our change observations
    this.unregisterChangeObservations();
    this.Super("destroy", arguments);
},

registerChangeObservations : function () {
    // To report dataSourceChanged, et. al., notifications we need to watch for changes
    // to the main editor and also to the session. The latter handles changes that
    // occur from the FieldEditor and RelationEditor.
    this.mainEditor.addMethods({
        itemChange : function (item, newValue, oldValue) {
            var dsEditor = this.creator;
            // If the DataSource ID is change, just report it
            if (item.name == "ID") {
                dsEditor.fireCallback(dsEditor.dataSourceChanged, "name", [null]);
            } else {
                // Only report the "first" change. The session will not have the change
                // yet on that first change. This prevents a change report on every edit.
                var name = this.getValue("ID");
                if (!dsEditor.hasPendingChanges(name)) {
                    dsEditor.fireCallback(dsEditor.dataSourceChanged, "name", [null]);
                }
            }
        }
    });

    var dsEditor = this;
    this.observe(this.session, "dataSourceChanged", function (name) {
        dsEditor.sessionDSChanged(name);
    });
    this.observe(this.session, "dataSourceRenamed", function (fromName,toName) {
        dsEditor.sessionDSRenamed(fromName,toName);
    });

    // We shouldn't need to update on add, and it's not clear how we'd handle a
    // delete of the dataSource we're currently editing.
},

// Notification methods when changes are made to the dataSource
// When the dataSource currently under edit is modified externally, update the editor to reflect
// the new values.

sessionDSChanged : function (name) {
    
    
    if (this._handlingSessionDSChanged) return;
    this._handlingSessionDSChanged = true;

    // Update the editor
    if (!this._pushingEditsToSession && name == this._currentDSName) {
        var ds = this.session.getDataSourceDefaults(name);
        this._startEditing(ds, true); // pass in reloaded parameter to prevent re-creating the edit node and losing existing metadata like _renamedFrom
    }

    // Fire the public notification
    // Report changes against current DS with a name==null to match the itemChange above.
    var dsEditor = this;
    var currentName = dsEditor.mainEditor.getValue("ID");
    if (currentName == name) name = null;
    dsEditor.fireCallback(dsEditor.dataSourceChanged, "name", [name]);
    delete this._handlingSessionDSChanged
},

sessionDSRenamed : function (fromName,toName) {
    if (this._handlingSessionDSRenamed) return;
    this._handlingSessionDSRenamed = true;
    
    if (!this._pushingEditsToSession && fromName == this._origDSName) {
        var ds = this.session.getDataSourceDefaults(toName); 
        this._startEditing(ds, true);

        // Remember the updated name so we can access details in the session
        this._currentDSName = toName
    }
    delete this._handlingSessionDSRenamed;
    
},

unregisterChangeObservations : function () {
    if (this.isObserving (this.session, "dataSourceChanged")) {
        this.ignore(this.session, "dataSourceChanged");
    }
    // There is nothing necessary to unhook the mainEditor.itemChange handler
},

execSQL : function () {
    var sql = this.deriveForm.getValue("sql");
    if (sql) {
        // strip whitespaces and trailing semicolons - these produce a syntax error when passed
        // to the JDBC tier
        sql = sql.trim().replace(/(.*);+/, "$1");
        var ds = isc.DataSource.get("DataSourceStore");
        ds.performCustomOperation("dsFromSQL", {dbName: this.mainEditor.getValue("dbName"), sql: sql}, this.getID()+".deriveDSLoaded(data)");
    }
},

deriveDSLoaded : function (data) {
    var ds = data.ds;
    this.dsLoaded(data.ds);
},

dsLoaded : function (dsConfig) {
    var ds = isc.DataSource.create(dsConfig);
    this.currentDS = ds;

    this.deriveFields(ds);
    this.previewGrid.setDataSource(ds);

    /* 
    var ob = ds.operationBindings;
    if (ob && ob.length > 0) {
        this.fetchOperationForm.setValues(ob[0]);
    }
    */
},

deriveFields : function (ds) {
    var fields = ds.getFieldNames();

    var newFields = [];
    for (var i = 0; i < fields.length; i++) {
        var fieldName = fields[i]
        var field = {};
        var dsField = ds.getField(fieldName);
        for (var key in dsField) {
            if (isc.isA.String(key) && key.startsWith("_")) continue;
            field[key] = dsField[key];
        }
        newFields.add(field);
    }

    var tree = isc.Tree.create({
        modelType: "parent",
        childrenProperty: "fields",
        titleProperty: "name",
        idField: "id",
	    nameProperty: "id",
        root: { id: 0, name: "root"},
        data: newFields
    });
    this.fieldEditor.setData(tree);
}

}); // end DataSourceEditor.addProperties

isc.DataSourceEditor.registerStringMethods({
    //> @method dataSourceEditor.editComplete
    // Method called by default from +link{saveClick} or +link{cancelClick} in response to a
    // user clicking the save or cancel buttons when +link{showActionButtons} is enabled.
    // <P>
    // It is the responsibility of the caller to respond accordingly to save any pending changes
    // and/or close or hide the editor.
    // <P>
    // Assuming the edit is not canceled, a typical handler will call +link{save} to have
    // pending changes saved to the +link{dsDataSource}. Individual DataSource saves can
    // be handled with +link{dataSourceSaved}.
    // <P>
    // If editing a single DataSource (like from +link{editSavedFromXml}), the updated XML can
    // be retrieved with +link{getDataSourceXml}.
    //
    // @param canceled (Boolean) true if the edit is being canceled (i.e. nothing to save)
    //
    // @visibility reifyOnSite
    //<
    editComplete: "canceled",

    //> @method dataSourceEditor.dataSourceSaved
    // Method called by +link{save} after each DataSource is saved.
    // <P>
    // There is no default implementation.
    //
    // @param name (Identifier) ID of the DataSource that was saved
    // @param ds (DataSource) Live instance of DataSource
    // @param isNew (Boolean) true if the DataSource is new
    // @param originalName (Identifier) the original DataSource ID if the DataSource was renamed
    //
    // @visibility reifyOnSite
    //<
    dataSourceSaved: "name,ds,isNew,originalName",

    //> @method dataSourceEditor.dataSourceChanged
    // Notification method fired when a DataSource is changed.
    //
    // @param name (String) the DataSource ID that changed. null for the currently editing one.
    // @param defaults (Properties) the updated DataSource defaults
    // @visibility reifyOnSite
    //<
    dataSourceChanged: "name"
});

/**************************************************
Parsed Data DataSource Editor
***************************************************/

//> @class ParsedDataDSEditor
// Provides a UI for creating +link{DataSource, DataSources}
// whose data comes from parsing an input file. A tab is included
// to show the input data as parsed based on field edits.
//
// @inheritsFrom VLayout
// @treeLocation Client Reference/Tools
// @visibility reifyOnSite
//<
isc.defineClass("ParsedDataDSEditor", "VLayout");


isc.ParsedDataDSEditor.addProperties({

    //> @attr parsedDataDSEditor.dsDataSource (DataSource | ID : null : IRW)
    // +link{dataSource} to be used to load and save ds.xml files, via
    // +link{group:fileSource,fileSource operations}.
    //
    // @visibility reifyOnSite
    //<

    //> @attr parsedDataDSEditor.knownDataSources (Array of PaletteNode : null : IR)
    // A list of all known DataSources, as +link{paletteNode,PaletteNodes}, to be used when editing
    // foreign keys, relations and includeFrom fields.
    // Each element of the array should include at least <code>ID</code> and <code>type</code>
    // properties. If <code>defaults</code> are not included, these will be loaded from
    // +link{dsDataSource}.
    //
    // @visibility reifyOnSite
    //<

    //> @attr parsedDataDSEditor.useLiveDataSources (Boolean : null : IR)
    // Unless set to <code>false</code> live DataSource instances will be used
    // as obtained from +link{dataSource.get}, otherwise, a temporary DataSource
    // will be created as needed from defaults that will not conflict with any
    // live instance.
    // <P>
    // This property is applied to a created +link{session}.
    // If a <code>session</code> is provided directly, this property is ignored.
    //
    // @visibility reifyOnSite
    //<

    //> @attr parsedDataDSEditor.session (DataSourceEditorSession : null : IR)
    // The editor "session" that maintains the pending DataSource changes. A new session
    // is automatically created during init if not provided.
    // <P>
    // When creating the initial session, the following DataSourceEditor properties
    // are passed to the new session:
    // <ul>
    //   <li>+link{dsDataSource}</li>
    //   <li>+link{knownDataSources} as +link{dataSourceEditorSession.dataSources}</li>
    //   <li>+link{useLiveDataSources}</li>
    // </ul>
    //
    // @visibility reifyOnSite
    //<

    //> @attr parsedDataDSEditor.dsProperties (Object : null : IR)
    // The properties that define the DataSource to be created.
    //
    // @visibility reifyOnSite
    //<

    //> @attr parsedDataDSEditor.fileType (ImportFormat : null : IR)
    // Type of the input data: csv, xml or json. "auto" format is not supported.
    //
    // @visibility reifyOnSite
    //<

    //> @attr parsedDataDSEditor.rawData (String | Array of String : null : IR)
    // The input data to process into sample data. For JSON or XML data, this is a single
    // string. For CSV data, the value can be a single string or an array of strings (lines).
    // When a single string is provided, the lines will be extracted by breaking the string
    // at carriage-returns, line feeds or both.
    //
    // @visibility reifyOnSite
    //<

    //> @attr parsedDataDSEditor.guessedRecords (Array of Record : null : IR)
    // Records that have been previously been guessed by a +link{SchemaGuesser}. If not
    // provided, a parser and guesser will be used to determine these.
    //
    // @visibility reifyOnSite
    //<

    // i18n messages
    //---------------------------------------------------------------------------------------

    //> @attr parsedDataDSEditor.instructions (HTMLString : "Edit detected fields and observe results for imported data" : IRW)
    // Instructions to be shown within the editor above the DataSource ID.
    //
    // @setter setInstructions
    // @group i18nMessages
    // @visibility reifyOnSite
    //<
    instructions: "Edit detected fields and observe results for imported data",

    //> @method parsedDataDSEditor.setInstructions
    // Sets +link{instructions} to a new value and shows the instructions section.
    // If set to <code>null</code> the instruction section will be hidden.
    //
    // @param [contents] (HTMLString) the instructions to show or null to hide section
    // @visibility reifyOnSite
    //<
    setInstructions : function (contents) {
        if (contents) {
            this.instructionsFlow.setContents(contents);
        } else { 
            this.instructionsFlow.hide();
        }
    },


    // internal components
    //---------------------------------------------------------------------------------------

    instructionsFlowDefaults: {
        _constructor: isc.HTMLFlow,
        width: "100%",
        height: 35,
        padding: 10,
        visibility: "hidden"
    },

    tabSetDefaults: {
        _constructor: isc.TabSet,
        width: "100%",
        height: "*",
        tabs: [
            { title: "Edit Fields" },
            { title: "View Data", pane: "autoChild:viewDataPane" }
        ]
    },

    buttonLayoutDefaults: {
        _constructor: "HLayout",
        height:42,
        layoutMargin:10,
        membersMargin:10,
        align: "right"
    },
    
    cancelButtonDefaults: {
        _constructor: "IButton",
        autoDraw: false,
        title: "Cancel",
        width: 100,
        autoParent: "buttonLayout",
        click: function() {
            this.creator.editorPane.cancelClick();
        }
    },
    
    saveButtonDefaults: {
        _constructor: "IButton",
        autoDraw: false,
        title: "Save",
        width: 100,
        autoParent: "buttonLayout",
        click: function() {
            this.creator.editorPane.saveClick();
        }
    },
    
    // DS Editor tab
    dsEditorPaneDefaults: {
        _constructor: isc.DataSourceEditor,
        width: "100%",
        height: "100%",
        canAddChildSchema: false,
        canEditChildSchema: false,
        showActionButtons: false,
        mainStackProperties: {
            _constructor: "TSectionStack"
        },
        mainEditorProperties: {
            _constructor: "TComponentEditor",
            formConstructor: isc.TComponentEditor
        },
        fieldLayoutProperties: {
            _constructor: "TLayout"
        },

        editComplete : function (canceled) {
            // Push event into the parsedDataDSEditor
            this.fireCallback(this.creator.editComplete, "canceled", arguments);
        },
        dataSourceSaved : function (name, ds, isNew, renamedFrom) {
            // Push callback into the parsedDataDSEditor
            this.fireCallback(this.creator.dataSourceSaved, "name, ds, isNew, renamedFrom", arguments);
        }
    },

    // View Data tab
    viewDataPaneDefaults: {
        _constructor: isc.SectionStack,
        width: "100%",
        height: "100%",
        visibilityMode: "multiple",
        sections: [
            { name: "records", title: "Records", showHeader: false, expanded: true,
                items: [ "autoChild:dataViewer" ]
            },
            { name: "errors", title: "Warnings/Errors", //expanded: false, hidden: true,
                items: [ "autoChild:errorViewer" ]
            }
        ]
    },

    dataViewerDefaults: {
        _constructor: isc.ListGrid,
        width: "100%",
        height: "100%"
    },

    errorViewerDefaults: {
        _constructor: isc.ListGrid,
        width: "100%",
        autoFitData: "vertical",
        autoFitMaxRecords: 10,
        defaultFields: [
            { name: "fieldName", title: "Field", width: 200 },
            { name: "message", title: "Message", width: "*" }
        ]
    }
});

isc.ParsedDataDSEditor.addMethods({

    initWidget : function () {
        this.Super("initWidget", arguments);

        this.addAutoChild("instructionsFlow", {
            contents: this.instructions,
            visibility: (this.instructions ? "inherit" : "hidden")
        });
        this.addAutoChild("tabSet");
        this.addAutoChild("buttonLayout");

        this.addAutoChild("cancelButton");
        this.addAutoChild("saveButton");

        var editorPaneProperties = isc.addProperties({}, this.dsEditorPaneProperties, {
            session: this.session
        });
        if (this.getUniqueDataSourceID) {
            editorPaneProperties.getUniqueDataSourceID = this.getUniqueDataSourceID;
        }
        
        this.editorPane = this.createAutoChild("dsEditorPane", editorPaneProperties);
        this.tabSet.setTabPane(0, this.editorPane);

        this.dataPane = this.createAutoChild("viewDataPane");
        this.tabSet.setTabPane(1, this.dataPane);

        if (!this.fileType) {
            this.logWarn("fileType not specified. Defaulting to 'csv'");
            this.fileType = "csv";
        } else if (this.fileType.toLowerCase() == "auto") {
            this.logWarn("fileType 'auto' is not supported. Defaulting to 'csv'");
            this.fileType = "csv";
        }
    },

    //> @method parsedDataDSEditor.editNew
    // Starts editing a new DataSource where <code>dataSource</code> specifies the starting
    // properties of the new DataSource.
    // <p>
    // When the user either saves or cancels the edit, +link{editComplete} is called.
    // The caller can then determine what to do with the changes, if any.
    // See +link{editComplete} for details.
    //
    // @param [dataSource] (PaletteNode) the dataSource to be edited
    // @visibility reifyOnSite
    //<
    editNew : function (dataSource) {
        // Make a copy of dsProperties to avoid changing caller's version
        this.dsProperties = isc.addProperties({}, dataSource.defaults);
        delete this.dsProperties.ID;

        var guessedRecords = this.guessedRecords,
            guessedFields,
            parseDetails
        ;

        if (!guessedRecords) {
            this._createParserAndGuesser();

            var parsedFields = this.parsedFields,
                parsedData = this.parsedData
            ;

            var guesser = this.guesser;
            guesser.fields = parsedFields;
            guessedFields = guesser.extractFieldsFrom(parsedData);
            guessedRecords = guesser.convertData(parsedData);
            
            parseDetails = guesser.parseDetails;
        }

        this._rebindDataViewer(null, guessedRecords, parseDetails);

        this.editorPane.editNew(dataSource);
    },
        
    editSaved : function (dataSource) {
        
        this.logWarn("editSaved not supported");
    },
    
    //> @method parsedDataDSEditor.save
    // @include dataSourceEditor.save
    //
    // @visibility reifyOnSite
    //<
    save : function (callback) {
        this.editorPane.save(callback);
    },

    //> @method parsedDataDSEditor.validate
    // @include dataSourceEditor.validate
    //
    // @visibility reifyOnSite
    //<
    validate : function () {
        return this.editorPane.validate();
    },

    //> @method parsedDataDSEditor.validateForSave
    // Performs a number of validation and user confirmations so that the current DataSource
    // can be saved.
    // <p>
    // See +link{dataSourceEditor.validateForSave} for details.
    //
    // @visibility reifyOnSite
    //<
    validateForSave : function (callback) {
        return this.editorPane.validateForSave(callback);
    },

    //> @method parsedDataDSEditor.getUniqueDataSourceID
    // Called to obtain a unique DataSource ID for a new DataSource. By default this
    // method always returns 'newDataSource'. Override to implement a custom algorithm.
    //
    // @param callback (Callback) Callback to fire when the unique ID is determined. Takes a single
    //                            parameter <code>ID</code> - the unique DataSource ID
    // @visibility reifyOnSite
    //<
    getUniqueDataSourceID : function (callback) {
        callback("newDataSource");
    },
    
    _createParserAndGuesser : function () {
        if (!this.parser) {
            this.parser = isc.FileParser.create({ hasHeaderLine: true });
        }
        if (!this.parsedData || !this.parsedFields) {
            var parser = this.parser,
                fileType = this.fileType.toLowerCase(),
                rawData = this.rawData
            ;
    
            // Perform initial parse to get fields
            if (fileType == "json" || fileType == "xml") {
                // XML data is pre-processed into JSON before getting here
                this.parsedData = parser.parseJsonData(rawData); 
            } else if (fileType == "csv") {
                this.parsedData = parser.parseCsvData(rawData);
            }
            this.parsedFields = parser.getFields();
        }
        if (!this.guesser) {
            var parsedFields = this.parser.getFields();
            this.guesser = isc.SchemaGuesser.create({ fields: parsedFields });
        }
    },

    fieldNameChanged : function (fromName, toName) {
        this._createParserAndGuesser();

        var parser = this.parser,
            fileType = this.fileType,
            rawData = this.rawData,
            fieldNames = parser.getFieldNames(),
            idx = fieldNames.indexOf(fromName)
        ;
        if (idx < 0) return;

        fieldNames[idx] = toName;
        this.parser.fieldNames = fieldNames;

        var parsedData = this.parsedData;
        if (fileType == "csv") {
            // Re-parse original data with new field name
            parsedData = parser.parseCsvData(rawData);
        } else {
            // Rename the field in data
            parsedData.forEach(function (record) {
                if (record[fromName] != null) {
                    record[toName] = record[fromName];
                    delete record[fromName];
                }
            });
        }
        this.parsedData = parsedData;
        this.parsedFields = parser.getFields();

        // Guess fields and convert data
        var guesser = this.guesser;
        guesser.fields = this.parsedFields;
        var guessedFields = guesser.extractFieldsFrom(parsedData),
            guessedRecords = guesser.convertData(parsedData)
        ;

        // Re-create testDS with updated fields/data
        this._rebindDataViewer(guessedFields, guessedRecords, guesser.parseDetails);
    },

    fieldTypeChanged : function (field) {
        this._createParserAndGuesser();

        // Updated parsed fields to apply type
        var parsedFields = this.parsedFields,
            parsedData = this.parsedData
        ;

        var parsedField = parsedFields.find("name", field.name);
        if (parsedField) {
            parsedField.type = field.type;

            var guesser = this.guesser;
            guesser.fields = parsedFields;
            var guessedFields = guesser.extractFieldsFrom(parsedData),
                guessedRecords = guesser.convertData(parsedData)
            ;

            // Re-create testDS with updated fields/data
            this._rebindDataViewer(guessedFields, guessedRecords, guesser.parseDetails);
        }
    },

    _rebindDataViewer : function (guessedFields, guessedRecords, parseDetails) {
        // Re-create testDS with updated fields/data
        if (this.testDS) this.testDS.destroy();

        var dsProperties = isc.addProperties({}, this.dsProperties, {
            clientOnly: true,
            testData: guessedRecords
        });
        if (guessedFields) dsProperties.fields = guessedFields;

        this.testDS = isc.DataSource.create(dsProperties);

        // Rebind grid
        this.dataViewer.setDataSource(this.testDS);
        this.dataViewer.fetchData();

        if (parseDetails && parseDetails.length > 0) {
            var _this = this;
            this.dataPane.expandSection(1, function () {
                // Make sure the errorViewer has been created
                _this.errorViewer.setData(parseDetails);
            });
        } else if (guessedFields) {
            this.dataPane.collapseSection(1);
        }
    }
}); // end ParsedDataDSEditor.addProperties

isc.ParsedDataDSEditor.registerStringMethods({
    //> @method parsedDataDSEditor.editComplete
    // Method called in response to a user clicking the save or cancel buttons.
    // <P>
    // It is the responsibility of the caller to respond accordingly to save any pending changes
    // and/or close or hide the editor.
    // <P>
    // Assuming the edit is not canceled, a typical handler will call +link{save} to have
    // pending changes saved to the +link{dsDataSource}. Individual DataSource saves can
    // be handled with +link{dataSourceSaved}.
    //
    // @param canceled (Boolean) true if the edit is being canceled (i.e. nothing to save)
    //
    // @visibility reifyOnSite
    //<
    editComplete: "canceled",

    //> @method parsedDataDSEditor.dataSourceSaved
    // Method called by +link{save} after each DataSource is saved.
    // <P>
    // There is no default implementation.
    //
    // @param name (Identifier) ID of the DataSource that was saved
    // @param ds (DataSource) Live instance of DataSource
    // @param isNew (Boolean) true if the DataSource is new
    // @param originalName (Identifier) the original DataSource ID if the DataSource was renamed
    //
    // @visibility reifyOnSite
    //<
    dataSourceSaved: "name,ds,isNew,originalName"
});

//> @class DataSourceEditorSession
// Provides a stateful, edit session for the +link{class:DataSourceEditor} and +link{class:RelationEditor}.
// <P>
// A DataSourceEditorSession tracks pending changes to one or more DataSources during editing. The session
// maintains a map of edited DataSources and their modified state, allowing changes to be saved together
// when editing is complete. A session can be shared across multiple editor instances, enabling workflows
// where the user edits multiple DataSources or their relations, then saves all changes at once.
// <P>
// The session requires a +link{dataSourceEditorSession.dsDataSource} to load and save DataSource XML. For most use cases, this
// should be a +link{group:fileSource,fileSource} DataSource that handles file operations. 
// For custom storage, set +link{dataSourceEditorSession.dsDataSourceIsFileSource}
// to false and configure +link{dataSourceEditorSession.dsDataSourceIDField}, +link{dataSourceEditorSession.dsDataSourceContentsField}, and
// +link{dataSourceEditorSession.dsDataSourcePKField} to map to your storage schema.
// <P>
// A session may be created explicitly and applied to the editor as its +link{DataSourceEditor.session}.
// If not provided, one will be created automatically by the editor with properties passed through from
// the editor configuration (+link{DataSourceEditor.dsDataSource}, +link{DataSourceEditor.knownDataSources},
// +link{DataSourceEditor.useLiveDataSources}).
// 
// @treeLocation Client Reference/Tools/DataSourceEditor
// @visibility reifyOnSite
//<
isc.defineClass("DataSourceEditorSession");

isc.DataSourceEditorSession.addProperties({

    //> @attr dataSourceEditorSession.dsDataSource (DataSource | ID : null : IRW)
    // The +link{dataSource} to be used to load and save dataSource XML. Each
    // record in this dataSource represents a dataSource that may be edited within
    // the dataSourceEditorSession.
    // <P>
    // If the provided dataSource is not a +link{group:fileSource,fileSource dataSource},
    // developers should set +link{dsDataSourceIsFileSource} to false. In this case the
    // dsDataSource is expected to have a field to hold the dataSource ID
    // (+link{dsDataSourceIDField}), a field to hold the dataSource XML
    // content (+link{dsDataSourceContentsField}) and a primary key field
    // to identify unique records (+link{dsDataSourcePKField}).
    // <smartclient>
    // <P>
    // For additional customization, developers may also override methods
    // to load and save dataSource xml. See +link{fetchDataSourceXml()} and
    // +link{saveDataSourceXml()}.
    // </smartclient> 
    // @visibility reifyOnSite
    //<

    //> @attr dataSourceEditorSession.dsDataSourceIsFileSource (boolean : true : IR)
    // Is the specified +link{dsDataSource} a +link{group:fileSource}?
    // @visibility reifyOnSite
    //<
    dsDataSourceIsFileSource:true,

    //> @attr dataSourceEditorSession.dsDataSourceIDField (String : null : IR)
    // When +link{dsDataSourceIsFileSource} is false, this attribute indicates which field
    // in the +link{dsDataSource} holds the ID for each dataSource.
    // @visibility reifyOnSite
    //<
    
    //> @attr dataSourceEditorSession.dsDataSourceContentsField (String : null : IR)
    // When +link{dsDataSourceIsFileSource} is false, this attribute indicates which field
    // in the +link{dsDataSource} holds the xml contents for each dataSource.
    // @visibility reifyOnSite
    //<
    
    //> @attr dataSourceEditorSession.dsDataSourcePKField (String : null : IR)
    // When +link{dsDataSourceIsFileSource} is false, this attribute should be set to 
    // the +link{dataSourceField.primaryKey,primary key field} for the dsDataSource.
    // @visibility reifyOnSite
    //<

    
    //> @attr dataSourceEditorSession.useLiveDataSources (Boolean : null : IR)
    // Determines whether +link{dataSourceEditorSession.get()} should return a
    // live DataSource as obtained from +link{dataSource.get()}.
    // <P>
    // If set to false, a temporary DataSource
    // will be created as needed from defaults that will not conflict with any
    // live instance.
    //
    // @visibility reifyOnSite
    //<

    //> @attr dataSourceEditorSession.dataSources (Array of PaletteNode : null : IRW)
    // A list of all DataSources, as +link{paletteNode,PaletteNodes}, that are used
    // in the project.
    // <P>
    // These are only required when an edited DataSource can have relations to other 
    // DataSources. Although each node is a +link{PaletteNode}, the only required properties
    // are ID, type and defaults.
    // <P>
    // These nodes will be updated while editing.
    // <P>
    // If the list is provided at init, the session may not be ready for use immediately.
    // See +link{setDataSources} for how to monitor for ready state.
    //
    // @setter setDataSources
    // @getter getDataSources
    // @visibility reifyOnSite
    //<

    //> @attr dataSourceEditorSession.useChangeTracking (Boolean : null : IR)
    // Should DataSource changes be tracked?
    //
    // @visibility internal
    //<

    //> @method dataSourceEditorSession.setDataSources
    // Sets +link{dataSources} to a new list. This operation resets the state of the session
    // clearing any pending DataSource changes.
    // <P>
    // Because the session is reset and defaults may need to be loaded, the session isn't
    // ready for use again until this completes. Use +link{isReady} or +link{waitForReady}
    // to monitor the ready state.
    //
    // @param dataSources (Array of PaletteNode) the list of DataSources
    // @param callback (Callback) Callback to fire when the new DataSource list has been
    //                            loaded.
    // @visibility reifyOnSite
    //<
    setDataSources : function (dataSources, callback) {
        this.dataSources = dataSources;

        // Session is not ready until we have loaded all defaults
        this._isReady = false;

        // Any stashed dsRecord PKs are now out of date
        this._clearDSRecordPrimaryKeys();

        // Build map of DataSources. Each entry is a copy of the editNode provided.
        // 
        this._dsMap = {};
        var _this = this;

        if (dataSources) {

            var dataSourcesToLoad = [],
                dataSourcesToConvert = []
            ;

            for (var i = 0; i < dataSources.length; i++) {
                var node = isc.addProperties({}, dataSources[i]);
                if (node.xml) {
                    // The defaults are just enough to load it but for editing we really
                    // want the full defaults that represent the DS defintion. Clear these
                    // partial defaults and load defaults from storage.
                    delete node.defaults;

                    dataSourcesToConvert.push(node);
                } else if (node.defaults && isc.getKeys(node.defaults).length <= 1 && node.loadData) {
                
                    // The defaults are just enough to load it but for editing we really
                    // want the full defaults that represent the DS defintion. Clear these
                    // partial defaults and load defaults from storage.
                    delete node.defaults;

                    dataSourcesToLoad.add(node.ID);
                }
                this._dsMap[node.ID] = node;
            }
            const session = this;
            if (dataSourcesToLoad.length > 0) {
                var loadDataSourcesCallback = function (defaultsMapping) {
                    for (var dsName in defaultsMapping) {
                        _this._dsMap[dsName].defaults = defaultsMapping[dsName];
                    }
                    session._isReady = true;
                    session._fireReadyCallbacks();
                    if (callback) callback();
                }                
                this.loadDataSourcesDefaults(dataSourcesToLoad, loadDataSourcesCallback);
                // Don't fire ready callbacks until the dataSources have been loaded.
                return;
            } else if (dataSourcesToConvert.length > 0) {
                // DataSource PaletteNodes containing XML that needs to be converted to defaults
                var convertDataSourcesCallback = function (defaultsMapping) {
                    for (var dsName in defaultsMapping) {
                        _this._dsMap[dsName].defaults = defaultsMapping[dsName];
                    }
                    session._isReady = true;
                    session._fireReadyCallbacks();
                    if (callback) callback();
                }
                this.convertDataSourcesDefaults(dataSourcesToConvert, convertDataSourcesCallback);
                // Don't fire ready callbacks until the dataSources have been loaded.
                return;
            }

        }
        if (this._isReady) this._fireReadyCallbacks();
        if (callback) callback();
    },

    // Helper to add a single dataSource paletteNode to our dataSources dynamically
    // This allows us DataSourceEditor to edit a single DS with no explicitly 
    // specified knownDataSources
    _addToDataSources : function (dataSource) {
        if (this._dsMap[dataSource.ID]) return;

        var node = isc.addProperties({}, dataSource);
        this._dsMap[dataSource.ID] = node;

        

    },

    //> @method dataSourceEditorSession.getDataSources
    // Returns the +link{paletteNode,PaletteNodes} of all DataSources that are part of this
    // edit session. This includes any added or renamed DataSources.
    //
    // @return (Array of PaletteNode) the list of DataSource nodes in the session
    //
    // @visibility reifyOnSite
    //<
    getDataSources : function () {
        var dataSources = [],
            map = this._dsMap
        ;
        for (var name in map) {
            dataSources.add(map[name]);
        }
        return dataSources;
    },

    //> @method dataSourceEditorSession.getAddedDataSources
    // Returns a list of all DataSource IDs in the session that have been added by +link{add}.
    //
    // @return (Array of Identifier) the list of added DataSource IDs
    //
    // @visibility reifyOnSite
    //<
    getAddedDataSources : function () {
        return this.getDataSources().filter(function (node) {
            return node._added;
        }).getProperty("ID");
    },

    //> @method dataSourceEditorSession.getUpdatedDataSources
    // Returns a list of all DataSource IDs in the session that have been updated by +link{set}.
    //
    // @return (Array of Identifier) the list of updated DataSource IDs
    //
    // @visibility reifyOnSite
    //<
    getUpdatedDataSources : function () {
        return this.getDataSources().filter(function (node) {
            return node._updated;
        }).getProperty("ID");
    },

    //> @method dataSourceEditorSession.getRenamedDataSources
    // Returns an object mapping original DataSource IDs to their new IDs as renamed by
    // +link{add}.
    //
    // @return (Object) mapping of IDs to renamed DataSources in the format
    //                  {originalName: newName, ...}
    //
    // @visibility reifyOnSite
    //<
    getRenamedDataSources : function () {
        var result = {};
        this.getDataSources().map(function (node) {
            if (node._renamedFrom != null) {                
                result[node._renamedFrom] = node.ID;
            }
        });
        return result;
    },

    //> @method dataSourceEditorSession.isUpdated
    // Returns true if the specified DataSource has been updated in this session (i.e.
    // has pending changes).
    //
    // @return (Boolean) true if the DataSource is updated in this session
    //
    // @visibility reifyOnSite
    //<
    isUpdated : function (name) {
        var node = this._dsMap[name];
        return (node && node._updated);
    },

    //> @method dataSourceEditorSession.getDataSourceDefaults
    // Returns the defaults object for a specified DataSource ID. These defaults may
    // have originated from an initial +link{dsDataSource} load or reflect updates made
    // by +link{add} or +link{set} calls.
    //
    // @return (Properties) defaults for specified DataSource ID
    //
    // @visibility reifyOnSite
    //<
    getDataSourceDefaults : function (name) {
        var node = this._dsMap[name];
        return (node ? node.defaults : null);
    },

    //> @method dataSourceEditorSession.getDataSourceXml
    // Returns the XML for a specified DataSource ID. The source defaults may
    // have originated from an initial +link{dsDataSource} load or reflect updates made
    // by +link{add} or +link{set} calls.
    //
    // @return (String) XML for specified DataSource ID
    //
    // @visibility reifyOnSite
    //<
    getDataSourceXml : function (name) {
        var node = this._dsMap[name],
            defaults = (node ? node.defaults : null)
        ;
        return defaults && this.serializeDataSource(defaults);
    },

    //> @method dataSourceEditorSession.get
    // Returns a live DataSource instance for the specified DataSource. When
    // +link{useLiveDataSources} is disabled, a temporary DataSource that doesn't conflict
    // with system live DataSources is returned. The same is returned in live DataSource
    // mode for a DataSource that has been added, updated or removed. Otherwise, a live
    // DataSource is returned from +link{DataSource.get}.
    // <P>
    // When the session is destroyed, any temporary DataSource instances are also destroyed.
    //
    // @param name (Identifier) the DataSource ID to get
    // @return (DataSource) the live DataSource instance
    //
    // @visibility reifyOnSite
    //<
    get : function (name) {
        if (!name) return null;

        var node = this._dsMap[name];
        if (node && (this.useLiveDataSources == false || node._updated || node._added || node._renamedFrom)) {
            var ds = node.liveObject;
            if (!ds) {
                node.liveObject = ds = this.createTempLiveDSInstance(node.defaults);
            }
            return ds;
        }
        return this.useLiveDataSources !== false && isc.DS.get(name);
    },

    //> @method dataSourceEditorSession.set
    // Sets the defaults object, as a copy, for a specified DataSource ID.
    // The DataSource must already exist within the session from either an +link{add} call
    // or +link{dataSources}.
    // <P>
    // If the provided defaults don't match the defaults already stored, the DataSource is
    // marked as 'updated' and will be included in the results of +link{getUpdatedDataSources}.
    //
    // @param name (Identifier) the DataSource ID
    // @param defaults (Properties) the DataSource defaults representing the DataSource
    // @param dontMarkUpdated (Boolean) when true, the DataSource will not be marked as 'updated'
    //
    // @visibility reifyOnSite
    //<
    set : function (name, defaults, dontMarkUpdated) {      
        
        var node = this._dsMap[name];
        if (!node) {
            this.logWarn("Cannot update DataSource '" + name + "' because it is not known " + this.getStackTrace());
            return;
        }
        var changed = !isc.Canvas.compareValues(node.defaults, defaults);
        if (changed) {
            // Save a copy so it doesn't get affected by the caller
            node.defaults = isc.clone(defaults);
            if (!node._added && node._renamedFrom == null && !dontMarkUpdated) {
                node._updated = true;
            }
            
            this.fireDataSourceChanged(node);
            
            // Drop cached instance
            var ds = node.liveObject;
            if (ds && !ds.addGlobalId) {
                ds.destroy();
            }
            node.liveObject = null;

            // Drop computed relations
            if (this.dsRelations) {
                this.dsRelations.reset();
            };
        }
    },

    //> @method dataSourceEditorSession.add
    // Adds a new DataSource to the session. The DataSource is marked as 'added' whether it
    // previously existed in the session or not and will be included in the results of
    // +link{getAddedDataSources}.
    //
    // @param name (Identifier) the DataSource ID
    // @param defaults (Properties) the DataSource defaults representing the DataSource
    // @param dontMarkAdded (Boolean) when true, the DataSource will not be marked as 'added'
    //
    // @visibility reifyOnSite
    //<
    add : function (name, defaults, dontMarkAdded, renamedFrom) {        
        
        this._dsMap[name] = {
            ID: name,
            // Save a copy so it doesn't get affected by the caller
            defaults: isc.clone(defaults)
        };
        if (!dontMarkAdded) {
            this._dsMap[name]._added = true;
            this.fireDataSourceAdded(this._dsMap[name]);
        }
        if (dontMarkAdded && renamedFrom) {
            this._dsMap[name]._renamedFrom = renamedFrom;
            this.fireDataSourceRenamed(this._dsMap[name]);
        }

        // Drop computed relations
        if (this.dsRelations) {
            this.dsRelations.reset();
        }
    },

    //> @method dataSourceEditorSession.rename
    // Renames DataSource in the session. The DataSource is marked as 'renamed' and will
    // be included in the results of +link{getRenamedDataSources}.
    //
    // @param existingName (Identifier) the DataSource ID to be renamed
    // @param newName (Identifier) the new (renamed) DataSource ID
    // @param defaults (Properties) the DataSource defaults representing the DataSource as renamed
    //
    // @visibility reifyOnSite
    //<
    rename : function (existingName, newName, defaults, added) {

        var node = this._dsMap[existingName];
        if (node == null) {
            this.logWarn("Unable to find existing dataSource with name:" + existingName);
            return;
        }

        // If the node has already been renamed once, remember the original ID
        var origName = node._renamedFrom || existingName;

        // Save defaults with the new ID marked as 'renamed'
        this.add(newName, defaults, !added, origName);
        // remove previous session defaults
        this.delete(existingName);
        
        // It's possible the DS was renamed temporarily in the session
        // In this case we're now doing a standard update again
        if (origName == newName) {
            var newNode = this._dsMap[newName];
            delete newNode._renamedFrom;
        }

    },

    //> @method dataSourceEditorSession.delete
    // Deletes the specified DataSource from the session. This is used to keep the session
    // in-sync with the caller.
    // <P>
    // Note that the DataSource will not be removed from any storage based on this call.
    //
    // @param name (Identifier) the DataSource ID
    //
    // @visibility reifyOnSite
    //<
    delete : function (name) {
        // Save the deleted DS node so it can be rolled back if needed
        if (!this._deletedDsMap) this._deletedDsMap = {};
        this._deletedDsMap[name] = this._dsMap[name];
        // Remove the DS from the session
        delete this._dsMap[name];
        this.fireDataSourceDeleted(name);
    },

    recordAddedField : function (name, field) {        
        var node = this._dsMap[name];
        if (!node) {
            this.logWarn("Cannot update DataSource '" + name + "' because it is not known");
            return;
        }
        var fieldName = field.name;
        // Don't record field addition against new or renamed DataSources
        if (!node._added && node._renamedFrom == null) {
            if (!node.addedFields) node.addedFields = [];
            if (!node.addedFields.contains(fieldName)) {
                node.addedFields.add(fieldName);
            }
        }

        this.addChange(name, "addField", {
            fieldName: fieldName,
            field: isc.clone(field)
        });
    },

    getAddedFields : function (name) {
        var node = this._dsMap[name];
        return node && node.addedFields;
    },

    // Updates fields and cacheData.  IncludeFrom fields in the same or
    // other DataSources may also be removed.
    removeField : function (name, fieldName, foreignKey, defaults) {
        var node = this._dsMap[name];
        if (!node) {
            this.logWarn("Cannot update DataSource '" + name + "' because it is not known");
            return;
        }
        this.recordRemovedField(name, fieldName);

        // Update defaults removing the field in question
        var updateSession = !defaults;
        defaults = defaults || isc.clone(this.getDataSourceDefaults(name));
        var fields = defaults.fields,
            field = fields.find("name", fieldName),
            data = defaults.cacheData
        ;

        var changeContext = this.addChange(name, "removeField", {
            fieldName: fieldName,
            field: isc.clone(field)
        });
        this.startChangeContext(changeContext);

        // Remove any includeFrom fields that reference this field, or if the field
        // is a foreignKey, remove includeFrom fields that use the relation
        if (foreignKey != null) {
            this.removeIncludeFieldsForForeignKey(name, foreignKey);
        } else {
            this.removeIncludeFieldsReferencingField(name, fieldName);
        }

        // Remove deleted field
        var fieldIndex = fields.findIndex("name", fieldName);
        if (fieldIndex >= 0) {
            fields.removeAt(fieldIndex);
            // Caller is responsible for rebinding the field editor
        }

        // Remove any displayField references to deleted field
        var referencedFields = fields.findAll("displayField", fieldName);
        if (referencedFields && referencedFields.length > 0) {
            referencedFields.map(function (field) {
                delete field.displayField;
            });
        }
        
        // Remove deleted field from sample data
        if (data) {
            data.forEach(function (record) {
                if (record[fieldName] != null) {
                    record[fieldName] = null;
                }
            });
            // Caller is responsible for rebinding the sample data editor
        }

        this.endChangeContext(changeContext);

        if (updateSession) {
            this.set(name, defaults);
        }

        // for convenience
        return defaults;
    },

    recordRemovedField : function (name, fieldName) {
        var node = this._dsMap[name];
        if (!node) {
            this.logWarn("Cannot update DataSource '" + name + "' because it is not known");
            return;
        }
        // Don't record field removal against new DataSources only
        if (!node._added) {
            if (node.addedFields && node.addedFields.contains(fieldName)) {
                // Field was just added so remove the indication of add
                node.addedFields.remove(fieldName);
            } else if (node.renamedFields && isc.getValues(node.renamedFields).contains(fieldName)) {
                // Field was previously renamed so instead of having the rename and the
                // delete, drop the rename and record the delete of the original field
                var fromName = node.renamedFields[fieldName];
                delete node.renamedFields[fieldName];
                fieldName = fromName;

                if (!node.removedFields) node.removedFields = [];
                if (!node.removedFields.contains(fieldName)) {
                    node.removedFields.add(fieldName);
                }
            } else {
                // Otherwise, record the removed field
                if (!node.removedFields) node.removedFields = [];
                if (!node.removedFields.contains(fieldName)) {
                    node.removedFields.add(fieldName);
                }
            }
        }
    },

    getRemovedFields : function (name) {
        var node = this._dsMap[name];
        return node && node.removedFields;
    },

    //> @method dataSourceEditorSession.standardizeMockDataSource
    // Convert a +link{MockDataSource} using +link{mockDatasource.mockData,"mock data"}
    // into a standard one with fields and sample data derived from the mock data.
    // <P>
    // The session defaults are updated in place as well as being returned for convenience.
    //
    // @param name (Identifier) the DataSource ID to convert
    // @return (Properties) defaults for the converted DataSource
    // @visibility reifyOnSite
    //<
    standardizeMockDataSource : function (name) {
        var ds = this.get(name);

        // DataSource instance has derived fields from the mockData. Use those as templates
        // for the new standardized DataSource
        var fieldNames = ds.getFieldNames(),
            fields = []
        ;
        for (var i = 0; i < fieldNames.length; i++) {
            var field = ds.getField(fieldNames[i]);
            field = fields[i] = isc.clone(field);
            // Validators that are present must be automatically generated ones and they
            // should not be exposed in the editor. In fact they will cause issues if the
            // field type is changed because they remain.
            delete field.validators;
            delete field._typeDefaultsAdded;
            delete field._simpleType;
            // If the field title is auto-derived, drop it so it will continue to 
            // be auto-derived if the name changes.
            if (field._titleAutoDerived) {
                delete field.title;
                delete field._titleAutoDerived;
            }
            // An emptyDisplayValue cannot be edited by the DataSourceEditor so clear it
            delete field.emptyDisplayValue;
        }

        // MockDataSource automatically adds a primaryKey of internalId. Re-process the
        // data to detect a primaryKey field within the data fields. If found, remove the
        // internalId field. Otherwise internalId stays.
        var mockData = this._getMockDataRecords(ds.mockData, ds.mockDataFormat),
            cacheData = ds.cacheData
        ;
        if (mockData) {
            var guesser = isc.SchemaGuesser.create({ detectPrimaryKey: true });
            guesser.dataSourceName = ds.ID;
            guesser.detectPrimaryKey = true;

            var guessedFields = guesser.extractFieldsFrom(mockData) || [],
                primaryKeyField = guessedFields.find("primaryKey", true)
            ;
            if (primaryKeyField) {
                var targetField = fields.find("name", primaryKeyField.name);
                if (targetField) {

                    // Remove current primary key field if it's not part of the guessed fields.
                    // That means it was added automatically.
                    var currentPrimaryKeyField = fields.find("primaryKey", true);
                    if (!guessedFields.find("name", currentPrimaryKeyField.name)) {
                        fields.remove(currentPrimaryKeyField);

                        // Remove previous PK values from data
                        var currentPrimaryKeyFieldName = currentPrimaryKeyField.name;
                        cacheData.forEach(function (record) {
                            if (record[currentPrimaryKeyFieldName] != null) {
                                delete record[currentPrimaryKeyFieldName];
                            }
                        });
                    }

                    var hasPK = fields.getProperty("primaryKey").or();
                    if (!hasPK) {
                        targetField.primaryKey = true;
                    }
                }
            }
        }

        // Update DS defaults to shift MDS from mockData to fields and cacheData
        var defaults = isc.clone(this.getDataSourceDefaults(name));

        defaults.fields = fields;
        defaults.cacheData = cacheData;
        delete defaults.mockData;
        delete defaults.mockDataFormat;
        delete defaults.mockDataPrimaryKey;

        // Save new, standardized DataSource defaults back into session
        this.set(name, defaults);

        // For convenience
        return defaults;
    },

    _getMockDataRecords : function (mockData, mockDataFormat) {
        if (mockData && isc.isA.String(mockData) && mockDataFormat != "mock") {
            // mockData provided as XML, CSV or JSON text. Convert data to
            // Array of Record.
            var parser = isc.FileParser.create({ hasHeaderLine: true });
            if (mockDataFormat == "xml") {
                // Process XML data into JSON.
                var xmlData = isc.xml.parseXML(mockData);
                if (!xmlData) {
                    this.logWarn("XML data in mockData could not be parsed");
                    return;
                }
                var elements = isc.xml.selectNodes(xmlData, "/"),
                    jsElements = isc.xml.toJS(elements)
                ;
                if (jsElements.length == 1) {
                    var encoder = isc.JSONEncoder.create({ dateFormat: "dateConstructor", prettyPrint: false });
                    var json = encoder.encode(jsElements[0]);
    
                    // XML data is now pre-processed into JSON
                    mockData = parser.parseJsonData(json, " loading pre-processed XML data for MockDataSource " + this.ID); 
                }
            } else if (mockDataFormat == "csv") {
                mockData = parser.parseCsvData(mockData); 
            } else if (mockDataFormat == "json") {
                mockData = parser.parseJsonData(mockData, " loading data for MockDataSource " + this.ID); 
            } else {
                this.logWarn("Invalid mockDataFormat '" + mockDataFormat + "'");
                return;
            }
            return mockData;
        }
    },

    removeIncludeFieldsForForeignKey : function (name, foreignKey, defaults) {
        var dotIndex = foreignKey.indexOf("."),
            // If there is no dot the foreignkey is a field within this dataSource.
            relatedDSName = (dotIndex == -1 ? name : foreignKey.substring(0, dotIndex)),
            dsPrefix = relatedDSName + "."
        ;

        // Get current defaults - just for reference to the fields
        defaults = defaults || this.getDataSourceDefaults(name);
        var fields = defaults.fields;
        for (var i = 0; i < fields.length; i++) {
            var field = fields[i];
            if (field.includeFrom && field.includeFrom.startsWith(dsPrefix)) {
                var includeFromFieldName = field.name;
                if (!includeFromFieldName) {
                    dotIndex = field.includeFrom.lastIndexOf(".");
                    includeFromFieldName = field.includeFrom.substring(dotIndex+1);
                }
                // Remove this includeFrom field because it uses the removed relation.
                // This can trigger cascading deletes of other referenced fields.
                this.removeField(name, includeFromFieldName, null, defaults);
            }
        }
        
    },

    removeIncludeFieldsReferencingField : function (name, fieldName) {
        var fieldReferences = this.getRelations().getFieldReferences(name, fieldName);
        if (fieldReferences) {
            for (var i = 0; i < fieldReferences.length; i++) {
                var ref = fieldReferences[i];
                // Remove the field that references the removed field.
                // This can trigger cascading deletes of other referenced fields.
                this.removeField(ref.dsId, ref.fieldName);
            }
        }
    },

    // Updates fields, validators and cacheData. IncludeFrom fields in the same or
    // other DataSources may also be affected.
    renameField : function (name, fromName, toName, defaults) {
        this.recordRenamedField(name, fromName, toName);

        // Update defaults renaming the field in question
        var updateSession = !defaults;
        defaults = defaults || isc.clone(this.getDataSourceDefaults(name));
        var fields = defaults.fields,
            data = defaults.cacheData
        ;

        // Rename field
        var field = fields.find("name", fromName);
        if (!field) return;
        field.name = toName;

        var changeContext = this.addChange(name, "changeFieldName", {
                fieldName: fromName,
                newName: toName
            });
        this.startChangeContext(changeContext);

        // Update any field validators that may reference the renamed field
        for (var i = 0; i < fields.length; i++) {
            var field = fields[i];
            if (field.validators && field.validators.length > 0) {
                var origValidators = isc.clone(field.validators),
                    validatorsChanged = false
                ;
                for (var j = 0; j < field.validators.length; j++) {
                    if (this.updateValidatorFieldNames(field.validators[j], fromName, toName)) {
                        validatorsChanged = true;
                    }
                }
                if (validatorsChanged) {
                    session.addChange(name, "changeFieldProperty", {
                        fieldName: field.name,
                        property: "validators",
                        originalValue: origValidators,
                        newValue: field.validators
                    });
                }
            }
        }

        // Rename field in sample data
        if (data) {
            data.forEach(function (record) {
                if (record[fromName] != null) {
                    record[toName] = record[fromName];
                    delete record[fromName];
                }
            });
        }

        if (updateSession) {
            this.set(name, defaults);
        }

        // Update any includeFrom values in current or related DataSources that refer
        // to this renamed field
        
        var session = this,
            fieldRenames = {},
            dsFieldsMap = {}
        ;
        fieldRenames[fromName] = toName;

        this.getDataSources().map(function (node) {
            var ds = session.get(node.ID);
            dsFieldsMap[node.ID] = ds.fields;
        });

        this.getRelations().updateIncludeFromReferences(name, fieldRenames, function (updatedDataSourceNames) {
            if (!updatedDataSourceNames) return;
            var includeFromSuffix = name + "." + toName;
            for (var i = 0; i < updatedDataSourceNames.length; i++) {
                var dsName = updatedDataSourceNames[i],
                    defs = session.getDataSourceDefaults(dsName),
                    fields = defs.fields
                ;
                for (var j = 0; j < fields.length; j++) {
                    var field = fields[j];
                    if (field.includeFrom) {
                        if (field.includeFrom.endsWith(includeFromSuffix) ||
                            (dsName == name && field.includeFrom == toName))
                        {
                            var oldField = dsFieldsMap[dsName][field.name];
                            session.addChange(dsName, "changeFieldProperty", {
                                fieldName: field.name,
                                property: "includeFrom",
                                originalValue: oldField.includeFrom,
                                newValue: field.includeFrom
                            });
                        }
                    }
                }
            }
        });

        this.endChangeContext(changeContext);

        // for convenience
        return defaults;
    },

    renameSampleDataField : function (fromName, toName, data) {
        data = data || this.getSampleData();
    
        // Rename field in records
        if (data) {
            data.forEach(function (record) {
                if (record[fromName] != null) {
                    record[toName] = record[fromName];
                    delete record[fromName];
                }
            });
        }
    
        return data;
    },
    
    recordRenamedField : function (name, fromName, toName) {
        var node = this._dsMap[name];
        if (!node) {
            this.logWarn("Cannot update DataSource '" + name + "' because it is not known");
            return;
        }
        // Don't record field rename against new or renamed DataSources
        if (!node._added && node._renamedFrom == null) {
            if (node.addedFields && node.addedFields.contains(fromName)) {
                // Field was just added so change the indication of add to the new name
                node.addedFields.remove(fromName);
                node.addedFields.add(toName);
            } else {
                // Otherwise, record the renamed field
                if (!node.renamedFields) node.renamedFields = {};

                var key = (fromName == null ? fromName : isc.getKeyForValue(fromName, node.renamedFields));
                if (key == toName) delete node.renamedFields[key];
                else node.renamedFields[key] = toName;
            }
        }
    },

    getRenamedFields : function (name) {
        var node = this._dsMap[name];
        return node && node.renamedFields;
    },

    updateValidatorFieldNames : function (validator, fromName, toName) {
        var applyWhen = validator.applyWhen;
        if (!applyWhen || isc.isA.emptyObject(applyWhen)) return;
        return this._replaceCriteriaFieldName(applyWhen, fromName, toName);
    },

    _replaceCriteriaFieldName : function (criteria, fromName, toName) {
        var operator = criteria.operator,
            changed = false
        ;
        if (operator == "and" || operator == "or") {
            var innerCriteria = criteria.criteria;
            for (var i = 0; i < innerCriteria.length; i++) {
                if (this._replaceCriteriaFieldName(innerCriteria[i], fromName, toName)) {
                    changed = true;
                }
            }
        } else {
            if (criteria.fieldName != null && criteria.fieldName == fromName) {
                criteria.fieldName = toName;
                changed = true;
            }
        }
        return changed;
    },

    changeFieldType : function (name, fieldName, newType, newLength, defaults) {
        var updateSession = !defaults;
        defaults = defaults || isc.clone(this.getDataSourceDefaults(name));
        var fields = defaults.fields,
            field = fields.find("name", fieldName),
            data = defaults.cacheData,
            typeChanged = (newType != field.type)
        ;

        var changeContext = this.addChange(name, "changeFieldType", {
                fieldName: fieldName,
                oldType: field.type,
                newType: newType,
                oldLength: field.length,
                // Length is only applicable to text fields (See +link{DataSourceField.length})
                newLength: (newType == "text" ? newLength : null)
            });
        this.startChangeContext(changeContext);

        // allowed values list is invalid when the type changes
        if (typeChanged && field.valueMap != null) {
            this.changeFieldProperty(name, fieldName, "valueMap", null, defaults);
            delete field.valueMap;
        }

        // Validators might be invalid for the new type. If the user
        // could edit the type of the field a prompt would be issued
        // to confirm changing and removing validators, however, since
        // this rename comes from a relation change we want to force
        // it to happen.
        if (typeChanged && field.validators != null) {
            this.changeFieldProperty(name, fieldName, "validators", null, defaults);
            delete field.validators;
        }

        // Update sample data field values based on new type
        if (typeChanged && data && data.length > 0) {
            var guesser = isc.SchemaGuesser.create({ fields: [ field ] }),
                revertedData = guesser.revertValues(data)
            ;

            // Apply new type to field
            field.type = newType;
            field.length = newLength;

            // Convert the raw values to correct types based on updated field
            guesser.fields = [ field ];
            guesser.extractFieldsFrom(revertedData);
            var convertedData = guesser.convertData(revertedData);

            // Copy converted data for changed field into sample data
            for (var i = 0; i < convertedData.length; i++) {
                data[i][fieldName] = convertedData[i][fieldName];
            }
        } else {
            // Apply new type to field
            field.type = newType;
            field.length = newLength;
        }

        this.endChangeContext(changeContext);

        if (updateSession) {
            this.set(name, defaults);
        }

        // for convenience
        return defaults;
    },

    changeFieldProperty : function (name, fieldName, propertyName, newValue, defaults) {
        var updateSession = !defaults;
        defaults = defaults || isc.clone(this.getDataSourceDefaults(name));
        var fields = defaults.fields,
            field = fields.find("name", fieldName)
        ;

        this.addChange(name, "changeFieldProperty", {
            fieldName: fieldName,
            property: propertyName,
            originalValue: field[propertyName],
            newValue: newValue
        });

        if (updateSession) {
            this.set(name, defaults);
        }

        // for convenience
        return defaults;
    },

    init : function () {
        this.Super("init", arguments);

        this._dsChanges = isc.Tree.create();
        this._dsChangeContext = [];

        // Initialize the DS map
        var self = this;
        this.setDataSources(this.dataSources, function () {
            self._isReady = true;
            self._fireReadyCallbacks();
        });
    },

    destroy : function () {
        // Destroy any cached temporary DataSources
        this.getDataSources().map(function (node) {
            // Drop cached instance
            var ds = node.liveObject;
            if (ds && !ds.addGlobalId) {
                ds.destroy();
            }
        });
        
        return this.Super("destroy", arguments);
    },

    //> @method dataSourceEditorSession.isReady
    // Is the session ready for use? Use +link{waitForReady} to wait for the session
    // to become ready.
    // <P>
    // A session is not ready for use until all known DataSources have loaded their
    // defaults.
    //
    // @return (Boolean) true if the session is ready
    //
    // @visibility reifyOnSite
    //<
    isReady : function () {
        return this._isReady;
    },

    //> @method dataSourceEditorSession.waitForReady
    // Wait for the session to be ready for use. To check the current ready status,
    // call +link{isReady}.
    // <P>
    // A session is not ready for use until all known DataSources have loaded their
    // defaults.
    //
    // @param callback (Callback) Callback to fire when the session becomes ready
    //
    // @visibility reifyOnSite
    //<
    waitForReady : function (callback) {
        if (this._isReady) {
            // Fire asynchronously for consistency
            this.delayFireCallback(callback);
            return;
        }
        if (this._readyCallbacks == null) this._readyCallbacks = [];
        this._readyCallbacks.push(callback);

    },
    _fireReadyCallbacks : function () {
        if (this._readyCallbacks) {
            var _this = this;
            this._readyCallbacks.forEach(function (callback) {
                _this.fireCallback(callback);
            });
            delete this._readyCallbacks;
        }
    },

    //> @method dataSourceEditorSession.getRelations
    // Returns a link{DSRelations} object for the session's DataSources.
    //
    // @return (DSRelations) the relations for the session DataSources
    //
    // @visibility internal
    //<
    getRelations : function () {
        var session = this;

        // Create DS list for use in the relations
        var dsList = this.getDataSources().map(function (node) {
            return session.get(node.ID);
        });

        if (!this.dsRelations) {
            this.dsRelations = isc.DSRelations.create({
                session: this,
                dataSources: dsList,

                

                // Hook DSRelations fetch/save operations to operate from the session
                getDataSourceDefaults : function (dsName, callback) {
                    // Clone defaults before returning so they can be updated in place
                    // without altering those in the session
                    var defaults = this.session.getDataSourceDefaults(dsName);
                    callback(dsName, isc.clone(defaults));
                },
                saveDataSource : function (defaults, callback) {
                    
                    this.session.set(defaults.ID, defaults);
                    if (callback) callback();
                },
                createLiveInstance : function (defaults) {
                    // nothing to do - instance will be created on demand
                },
                // RelationEditor needs additional post-processing for relations.
                // This will not affect DataSourceEditor so it can be done for both.
                getRelationsForDataSource : function (name) {
                    var relations = this.Super("getRelationsForDataSource", arguments),
                        defaults = this.session.getDataSourceDefaults(name)
                    ;

                    if (defaults) {
                        for (var i = 0; i < relations.length; i++) {
                            var relation = relations[i],
                                type = relation.type,
                                displayField = relation.displayField,
                                includeField
                            ;
                            // Only post-process direct and indirect FK relations
                            if (type == "Self") continue;
                
                            // The FK displayField references the includeFrom field by name but
                            // that name is either the includeFrom field name or an explicit name
                            // on the field (Name as). For editing the displayField should be the
                            // actual field name on the related DS and includeField is the name
                            // of the included field, if renamed. 
                            if (displayField && name == defaults.ID) {
                                var field = (type == "M-1" ?
                                                defaults.fields.find("name", displayField) :
                                                this.session.get(relation.dsId).getField(displayField));
                                if (field && field.includeFrom) {
                                    includeField = displayField;
                                    var split = field.includeFrom.split(".");
                                    if (split && split.length >= 2) {
                                        displayField = split[split.length-1];
                                    } else {
                                        displayField = field.includeFrom;
                                    }
                                    if (displayField == includeField) includeField = null;
                                }
                                relation.displayField = displayField;
                                relation.includeField = includeField;
                            }
                        }
                    }
                    return relations;
                }
            });
        } else {
            this.dsRelations.setDataSources(dsList);
        }
        return this.dsRelations;
    },

    //> @method dataSourceEditorSession.saveChanges
    // Saves all changes made to session to +link{dsDataSource} calling
    // <code>dataSourceSavedCallback</code> after each DataSource is saved and finally calling
    // <code>completeCallback</code> when done.
    // <P>
    // This method will save updated dataSources, renamed dataSources and newly added dataSources
    // by invoking +link{saveUpdatedDataSourceDefaults()}, +link{saveRenamedDataSourceDefaults()}, 
    // and +link{saveAddedDataSourceDefaults()} respectively.
    //
    // @param [completeCallback] (Callback) the callback to be called when all saves are complete
    // @param [dataSourceSavedCallback] (Callback) the callback to be called after each DataSource
    //               is saved. Has the same signature as +link{dataSourceEditor.dataSourceSaved}.
    //
    // @visibility reifyOnSite
    //<
    saveChanges : function (completeCallback, dataSourceSavedCallback) {
        var session = this;

        var saveRenamedDataSources = function (callback) {
            var renamedDataSources = session.getRenamedDataSources(),
                count = isc.getKeys(renamedDataSources).length
            ;
            if (count == 0) {
                return callback(true);
            };
            var dsActionLoopSuccess = true;
            for (var key in renamedDataSources) {
                var dsName = renamedDataSources[key],
                    origName = key,
                    defaults = session.getDataSourceDefaults(dsName)
                ;

                session.saveRenamedDataSourceDefaults(dsName, origName, defaults, function (success) {
                    dsActionLoopSuccess = dsActionLoopSuccess && success;
                    
                    if (session.useLiveDataSources != false) {
                        // Reload the new DS
                        isc.DataSource.load(dsName, function () {
                            var ds = isc.DS.get(dsName);
                            if (success) session.fireCallback(dataSourceSavedCallback, "name, ds, isNew, renamedFrom", [dsName, ds, null, origName]);
                            if (--count == 0) {
                                callback(dsActionLoopSuccess);
                            }
                        }, true, true);
                    } else {
                        if (success) session.fireCallback(dataSourceSavedCallback, "name, ds, isNew, renamedFrom", [dsName, null, null, origName]);
                        if (--count == 0) {
                            callback(dsActionLoopSuccess);
                        }
                    }
                });
            }
        };

        var saveAddedDataSources = function (callback) {
            var addedDataSources = session.getAddedDataSources(),
                count = addedDataSources.length
            ;
            if (count == 0) {
                return callback(true);
            }
            var dsActionLoopSuccess = true;
            for (var i = 0; i < addedDataSources.length; i++) {
                var dsName = addedDataSources[i],
                    defaults = session.getDataSourceDefaults(dsName)
                ;
                session.saveAddedDataSourceDefaults(dsName, defaults, function (success) {
                    dsActionLoopSuccess = dsActionLoopSuccess && success;
                    if (session.useLiveDataSources != false) {
                        // Reload the new DS
                        isc.DataSource.load(dsName, function () {
                            var ds = isc.DS.get(dsName);
                            if (success) session.fireCallback(dataSourceSavedCallback, "name, ds, isNew, renamedFrom", [dsName, ds, true]);
                            if (--count == 0) {
                                callback(dsActionLoopSuccess);
                            }
                        }, true, true);
                    } else {
                        if (success) session.fireCallback(dataSourceSavedCallback, "name, ds, isNew, renamedFrom", [dsName, null, true]);
                        if (--count == 0) {
                            callback(dsActionLoopSuccess);
                        }
                    }
                });
            }
        };

        var saveUpdatedDataSources = function (callback) {
            var updatedDataSources = session.getUpdatedDataSources(),
                count = updatedDataSources.length
            ;
            if (count == 0) {
                return callback(true);
            }
            var dsActionLoopSuccess = true;
            for (var i = 0; i < updatedDataSources.length; i++) {
                var dsName = updatedDataSources[i],
                    defaults = session.getDataSourceDefaults(dsName)
                ;
                session.saveUpdatedDataSourceDefaults(dsName, defaults, function (success) {
                    dsActionLoopSuccess = dsActionLoopSuccess && success;
                    if (session.useLiveDataSources != false) {
                        // Reload the new DS
                        isc.DataSource.load(dsName, function () {
                            var ds = isc.DS.get(dsName);
                            if (success) session.fireCallback(dataSourceSavedCallback, "name, ds, isNew, renamedFrom", [dsName, ds]);
                            if (--count == 0) {
                                callback(dsActionLoopSuccess);
                            }
                        }, true, true);
                    } else {
                        if (success) session.fireCallback(dataSourceSavedCallback, "name, ds, isNew, renamedFrom", [dsName]);
                        if (--count == 0) {
                            callback(dsActionLoopSuccess);
                        }
                    }
                });
            }
        };

        // Save pending DataSources        
        var _this = this;
        saveRenamedDataSources(function (renameSuccess) {
            saveAddedDataSources(function (addSuccess) {
                saveUpdatedDataSources(function (updateSuccess) {
                    var finalSuccess = renameSuccess && addSuccess && updateSuccess;
                    if (finalSuccess) {                        
                        _this.acceptChanges();
                    }

                    if (completeCallback) completeCallback(renameSuccess && addSuccess && updateSuccess);
                });
            });
        });
    },

    // If the target dataSource is not a filesource, we need the primary key 
    // value for each dataSource's record so we can update existing records.
    // We'll set this up lazily when fetching dataSource records.
    _cacheDSRecordPrimaryKey : function (dsName, primaryKey) {
        if (primaryKey == null) return;

        if (this._dsRecordMap == null) {
            this._dsRecordMap = {};
        }
        this._dsRecordMap[dsName] = primaryKey
    },
    _getDSRecordPrimaryKey : function (dsName) {
        return this._dsRecordMap && this._dsRecordMap[dsName];
    },

    _clearDSRecordPrimaryKey : function (dsName) {
        if (this._dsRecordMap) delete this._dsRecordMap[dsName];
    },
    _clearDSRecordPrimaryKeys : function () {
        delete this._dsRecordMap;
    },

    //> @method dataSourceEditorSession.saveRenamedDataSourceDefaults()
    // Method to save changes to a dataSource defaults when the user has changed the dataSource ID.
    // <P>
    // When permanent storage of dataSources is indexed by the dataSource ID, as with a 
    // +link{dsDataSourceIsFileSource,fileSource dataSource}, this will invoke +link{saveDataSourceDefaults()} to save the
    // renamed dataSource, followed by +link{removeDataSourceXml()} to remove the previously named
    // version.
    // <P>
    // If the dsDataSource does not index dataSources by ID, this method will simply fall through to 
    // +link{saveDataSourceDefaults()} to handle the update directly.
    // <P>
    // This method is invoked as part of the +link{saveChanges()} flow where appropriate
    //
    // @param dsName (String) new dataSource ID
    // @param origName (String) previous ID for the dataSource
    // @param defaults (Object) new defaults block
    // @param callback (Callback) callback to fire on completion. Takes a single boolean argument "success".
    // @visibility reifyOnSite
    //<
    saveRenamedDataSourceDefaults : function (dsName, origName, defaults, callback) {
        var updateOnly = this._shouldRenameViaUpdate();
        var session = this;
        if (updateOnly) {
            // Re-key our stored dsConfig record by the new name.
            var pk = this._getDSRecordPrimaryKey(origName);
            this._cacheDSRecordPrimaryKey(dsName, pk);

            // Allow the standard save flow to update the dsDataSource record with the new
            // name.     
            this.saveDataSourceDefaults(defaults, callback);

        } else {
            session.saveDataSourceDefaults(defaults, function (success) {
                if (!success) callback(false);
                else {
                    // Load original DS XML
                    session.fetchDataSourceXml(origName, function (xml) {
                        // Delete the original DS
                        session.deleteRenamedDataSource(
                            origName, xml, 
                            function (dsName, success) {
                                callback(success);
                            }
                        );
                    });
                }
            });
        }
    },

    // When saving a renamed dataSource, do we need to perform a remove and add or can we perform a simple update
    // to change the ID. If we're keying off the dataSource ID we need a remove / add
    _shouldRenameViaUpdate : function () {
        return !this.dsDataSourceIsFileSource || (this.dsDataSourcePKField != this.dsDataSourceIDField);
    },

    //> @method dataSourceEditorSession.saveAddedDataSourceDefaults()
    // Method to save a newly added dataSource to permanent storage.
    // <P>
    // Default implementation will call +link{saveDataSourceDefaults()} to save the
    // newly added dataSource
    // <P>
    // This method is invoked as part of the +link{saveChanges()} flow where appropriate
    //
    // @param dsName (String) new dataSource ID
    // @param defaults (Object) new defaults block
    // @param callback (Callback) callback to fire on completion. Takes a single boolean argument "success".
    // @visibility reifyOnSite
    //<
    saveAddedDataSourceDefaults : function (dsName, defaults, callback) {
        this.saveDataSourceDefaults(defaults, callback);
    },

    //> @method dataSourceEditorSession.saveUpdatedDataSourceDefaults()
    // Method to save updates to an existing dataSource to permanent storage. Note that this method
    // will only be called if the name has not changed - if a rename occurred, +link{saveRenamedDataSourceDefaults()}
    // will be invoked instead.
    // <P>
    // Default implementation will call +link{saveDataSourceDefaults()} to save the
    // updated dataSource
    // <P>
    // This method is invoked as part of the +link{saveChanges()} flow where appropriate
    //
    // @param dsName (String) new dataSource ID
    // @param defaults (Object) new defaults block
    // @param callback (Callback) callback to fire on completion. Takes a single boolean argument "success".
    // @visibility reifyOnSite
    //<
    saveUpdatedDataSourceDefaults : function (dsName, defaults, callback) {
        this.saveDataSourceDefaults(defaults, callback);
    },

    //> @method dataSourceEditorSession.acceptChanges
    // Clears session changes resetting the state of the session as if it was just loaded.
    // All current changes remain in place - only the report of changes is removed. For
    // example, +link{isUpdated} and +link{getUpdatedDataSources} will not report anything.
    // <P>
    // This method is useful after saving changes with +link{saveChanges} and then continuing
    // to use the same session for additional work.
    // <P>
    // To actually revert session contents to its original loaded state, including contents of
    // DataSources, use +link{rollbackChanges()}.
    //
    // @visibility reifyOnSite
    //<
    acceptChanges : function () {
        this.getDataSources().map(function (node) {
            delete node._added;
            delete node._updated;
            delete node._renamedFrom;
            delete node.addedFields;
            delete node.removedFields;
            delete node.renamedFields;
        });
        delete this._deletedDsMap;

        // Assertion: The dsRecord primary key cache doesn't need to be modified here. It should
        // already reflect the correct values for the (successfully saved) records.
    },

    //> @method dataSourceEditorSession.rollbackChanges
    // Revert session changes to their original loaded state. This includes reloading changed DataSource
    // definitions from storage. Doing so will cause the session to transition to not +link{isReady()}
    // state while reloading and any +link{waitForReady} callbacks will be fired once the reloads are
    // complete. Immediately thereafter the method callback, if provided, is called.
    // <P>
    // If only the edit state of the session should be cleared maintaining the current edits, see
    // +link{acceptChanges}.
    //
    // @param callback (Callback) callback to fire on completion
    // @visibility reifyOnSite
    //<
    rollbackChanges : function (callback) {
        // Session is not ready until we have loaded all defaults
        this._isReady = false;

        
        isc.RPC.startQueue();

        // Re-load any renamed/updated DataSource defaults
        var session = this;
        this.getDataSources().map(function (node) {
            if (node._updated || node._renamedFrom) {
                if (node._renamedFrom) {
                    // Move renamed node back to the original mapping
                    session._dsMap[node._renamedFrom] = node;
                    delete session._dsMap[node.ID];
                    // And update the node ID to original
                    node.ID = node._renamedFrom;

                    // Drop cached liveObject instance
                    var ds = node.liveObject;
                    if (ds && !ds.addGlobalId) {
                        ds.destroy();
                    }
                    node.liveObject = null;

                    // Drop the cached dsRecordPrimaryKey for the new-name'd ds
                    // if we have one.
                    // We'll lazily populate the old dsName record primary key
                    // as part of loadDataSourceDefaults()
                    session._clearDSRecordPrimaryKey(node.ID);

                }
                // Load stored defaults
                (function(node) {
                    session.loadDataSourceDefaults(node.ID, function (defaults) {
                        node.defaults = defaults;
                    });
                }(node));
            } else if (node._added) {
                // Drop added DataSources
                delete session._dsMap[node.ID];
            }
        });

        // Restore any deleted DataSources
        if (this._deletedDsMap) {
            for (var deletedName in this._deletedDsMap) {
                var node = this._deletedDsMap[deletedName];

                // Restore DS node into session
                this._dsMap[deletedName] = node;

                // Drop cached liveObject instance
                var ds = node.liveObject;
                if (ds && !ds.addGlobalId) {
                    ds.destroy();
                }
                node.liveObject = null;

                // Load stored defaults
                (function(node) {
                    session.loadDataSourceDefaults(node.ID, function (defaults) {
                        node.defaults = defaults;
                    });
                }(node));
            }
            delete this._deletedDsMap;
        }

        if (isc.RPC.getQueueTransactionId() != null) {
             isc.RPC.sendQueue(function () {
                // Clear editing state
                session.acceptChanges();

                // Drop computed relations
                if (session.dsRelations) {
                    session.dsRelations.reset();
                };
                session._isReady = true;
                session._fireReadyCallbacks();
                if (callback) callback();
            });
            // Callback will be triggered after queue is complete
            return;
        }
        // If we started a queue but didn't add any transactions to it, cancel it now
        
        isc.RPC.startQueue(false);

        // Clear editing state
        this.acceptChanges();

        this._isReady = true;
        this._fireReadyCallbacks();
        if (callback) callback();
    },

    //> @method dataSourceEditorSession.saveDataSourceDefaults
    // Save DataSource defaults by serializing them to Xml and delegating to
    // +link{saveDataSourceXml}. 
    // <P>
    // This method is invoked as part of the +link{saveChanges()} flow where appropriate
    //
    // @param defaults (Properties) the DataSource defaults to save
    // @param callback (Callback) Callback to fire when the defaults have been saved.
    //                            Takes a single parameter <code>success</code> - true
    //                            if save was successful.
    // @visibility reifyOnSite
    //<
    saveDataSourceDefaults : function (defaults, callback, extraFileSpec) {
        var xml = this.serializeDataSource(defaults);
    
        this.saveDataSourceXml(defaults.ID, xml, callback, extraFileSpec);
    },

    //> @method dataSourceEditorSession.fetchDataSourceXml
    // Retrieve DataSource Xml from persistent storage via a fetch against the +link{dsDataSource}.
    // <smartclient>
    // <P>
    // This method may be overridden to use custom logic to retrieve dataSource XML.
    // </smartclient>
    //
    // @param dsName (Identifier) the name of the DataSource to fetch
    // @param callback (Callback) Callback to fire when the source has been fetched.
    //                            Takes a single parameter <code>xml</code> - the fetched Xml.
    // @visibility reifyOnSite
    //<
    fetchDataSourceXml : function (dsName, callback) {

        if (this.dsDataSourceIsFileSource) {
            this.dsDataSource.getFile({
                fileName: dsName,
                fileType: "ds",
                fileFormat: "xml"
            }, function (dsResponse, data, dsRequest) {
                callback(data);
            }, {
                // DataSources are always shared across users
                operationId: "allOwners"
            });
        } else {

            // Data is stored in a normal DS - find the XML from the associated record
            var dsCriterion = {};
            dsCriterion[this.dsDataSourceIDField] = dsName;
            var _this = this;
            this.dsDataSource.fetchData(
                dsCriterion,
                function (dsResponse, data, dsRequest) {
                    if (!data || data.length == 0) {
                        isc.logWarn("Unable to retrieve dataSource XML for dataSource:" + dsName);
                        return;
                    }
                    var dsRecord = data[0];
                    // Remember the primary key for saving
                    _this._cacheDSRecordPrimaryKey(dsName, dsRecord[_this.dsDataSourcePKField]);

                    callback(dsRecord[_this.dsDataSourceContentsField]);
                }
            );
        }
    },

    //> @method dataSourceEditorSession.fetchDataSourceJS
    // Fetches DataSource Xml from persistent storage and converts it to JavaScript.
    // By default, the source is retrieved from the +link{dsDataSource}.
    // <smartclient>
    // <P>
    // This method may be overridden to use custom logic to retrieve dataSource JavaScript.
    // </smartclient>
    //
    // @param dsName (Identifier) the name of the DataSource to fetch
    // @param callback (Callback) Callback to fire when the source has been fetched.
    //                            Takes a single parameter <code>js</code> - the fetched JS.
    // @visibility reifyOnSite
    //<
    // Note: fetchDataSourceXML and fetchDataSourceJS are separate methods because the server
    // is required to load the original XML and to convert that XML to JavaScript in a 
    // schema-specified type-aware way.
    fetchDataSourceJS : function (dsName, callback) {
        if (this.dsDataSourceIsFileSource) {
            this.dsDataSource.getFile({
                fileName: dsName,
                fileType: "ds",
                fileFormat: "xml"
            }, function (dsResponse, data, dsRequest) {
                // data argument is the XML response. We want the JS response
                callback(dsResponse.data && dsResponse.data.length > 0 ? dsResponse.data[0].fileContentsJS : null);
            }, {
                // DataSources are always shared across users
                // and we want JavaScript in the response to avoid an extra xmlToJs call
                operationId: "allOwnersXmlToJs"
            });
        } else {
            var _this = this;
            this.fetchDataSourceXml(dsName, function (xml) {
                _this.dataSourceXMLToJS(xml, callback);
            });    
        }
    },

    //> @method dataSourceEditorSession.dataSourceXmlToDefaults()
    // Utility method to convert DataSource XML into a JavaScript defaults object.
    //
    // @param xml (String) DataSource XML
    // @param (Callback) Callback to invoke when the conversion is complete. Takes the defaults object as a 
    //  single argument or <code>null</code> if the conversion failed.
    //  
    // @visibility reifyOnSite
    //<
    dataSourceXmlToDefaults : function (xml, callback) {
        isc.DMI.callBuiltin({
            methodName: "xmlToJS",
            "arguments": xml,
            callback : function (rpcResponse, data, rpcRequest) {
                if (!data) {
                    callback(null);
                    return;
                }
                callback(isc.DataSourceEditor.extractDSDefaultsFromJS(data));
            }
        });
    },

    //> @method dataSourceEditorSession.dataSourceXmlToJS()
    // Utility method to convert DataSource XML into a JavaScript snippet defining the DataSource
    //
    // @param xml (String) DataSource XML
    // @param (Callback) Callback to invoke when the conversion is complete. Takes the converted JavaScript string as a 
    //  single argument, or <code>null</code> if the conversion failed.
    //  
    // @visibility reifyOnSite
    //<
    dataSourceXMLToJS : function (xml, callback) {
        isc.DMI.callBuiltin({
            methodName: "xmlToJS",
            "arguments": xml,
            callback: function (rpcResponse, data, rpcRequest) {
                if (!data) {
                    callback(null);
                    return;
                }
                callback(data);
            }
        });
    },

    //> @method dataSourceEditorSession.saveDataSourceXml
    // Save DataSource Xml to persistent storage. By default, this is to the
    // +link{dsDataSource}. 
    // <smartclient>
    // <P>
    // This method may be overridden to use custom logic to store dataSource XML.
    // </smartclient>
    //
    // @param dsName (Identifier) the name of the DataSource to save
    // @param xml (String) the DataSource definition in Xml
    // @param callback (Callback) Callback to fire when the Xml has been saved.
    //                            Takes a single parameter <code>success</code> - true
    //                            if save was successful.
    // @visibility reifyOnSite
    //<
    saveDataSourceXml : function (dsName, xml, callback, extraFileSpec) {

        if (this.dsDataSourceIsFileSource) {
            var fileSpec = isc.addProperties({}, {
                fileName: dsName,
                fileType: "ds",
                fileFormat: "xml"
            }, extraFileSpec)

            this.dsDataSource.saveFile(fileSpec, xml, function(resp, data) {
                var success = true;
                if (resp.status < 0) {
                    isc.warn(resp.data);
                    success = false;
                }
                if (callback) callback(success);
            }, {
                // DataSources are always shared across users - check for existing file to
                // overwrite without regard to ownerId
                operationId: "allOwners"
            });

        // Non-fileSource dataSource - perform an add or update
        } else {
                        
            // Save to the standard dataSource.        
            var pk = this._getDSRecordPrimaryKey(dsName);

            // It's an add if we don't have a primary key for the dataSource record
            var isEdit = (pk != null);

            var newRecord = {};
            newRecord[this.dsDataSourceIDField] = dsName;
            newRecord[this.dsDataSourceContentsField] = xml;

            if (isEdit) {
                newRecord[this.dsDataSourcePKField] = pk;
            } else {
                // Opportunity for custom record transform beyond the standard pk, ID and contents fields
                
                if (this.transformNewDSRecord) this.transformNewDSRecord(newRecord);
            }

            var _this = this;
            var saveCallback = function (dsResponse, data, dsRequest) {

                if (dsResponse.status == 0 && data.length == 1) {
                    if (!isEdit) {
                        var dsRecord = data[0];
                        _this._cacheDSRecordPrimaryKey(dsName, dsRecord[_this.dsDataSourcePKField]);
                    }
                }

                callback(dsResponse.status==0);
            };
            
            if (isEdit) {
                this.dsDataSource.updateData(
                    newRecord,
                    saveCallback
                );
            } else {
                this.dsDataSource.addData(
                    newRecord,
                    saveCallback
                );
            }
        }
    },

    //> @method dataSourceEditorSession.removeDataSourceXml
    // Remove DataSource Xml from persistent storage. When using a +link{dataSource.hasFile,fileSource},
    // this will issue a +link{dataSource.removeFile()} command.
    // <P>
    // This method will be invoked automatically as part of the +link{saveRenamedDataSourceDefaults()}
    // flow when the dataSource is indexed by the fileName in permanents storage as in a fileSource
    // dataSource.
    // <smartclient>
    // <P>
    // This method may be overridden to use custom logic to remove dataSource XML from
    // permanent storage.
    // </smartclient>
    //
    // @param dsName (Identifier) the name of the DataSource to save
    // @param callback (Callback) Callback to fire when the Xml has been removed
    // @visibility reifyOnSite
    //<
    removeDataSourceXml : function (dsName, callback) {
        if (this.dsDataSourceIsFileSource) {
            this.dsDataSource.removeFile({
                fileName: dsName,
                fileType: "ds",
                fileFormat: "xml"
            }, function (dsResponse, data, dsRequest) {
                var success = data != null && dsResponse.status == 0;
                callback(success);
            });
        } else {
            // Remove the record from the dataSource
            
            var pk = this._getDSRecordPrimaryKey(dsName);
            if (pk != null) {
                var pkMap = {};
                pkMap[this.dsDataSourcePKField] = pk;
                this.dsDataSource.removeData(
                    pkMap,
                    function (dsResponse, data, dsRequest) {
                        var success = data != null && dsResponse.status == 0;
                        callback(success);
                    }                    
                );
            }
        }
    },

    //> @method dataSourceEditorSession.hasDataSourceXml
    // Confirms if the DataSource has Xml contents in persistent storage.
    // By default, the existence is confirmed against the +link{dsDataSource}.
    // <smartclient>
    // <P>
    // This method may be overridden to use custom logic to check for the existence
    // of dataSource XML in permanent storage.
    // </smartclient>
    //
    // @param dsName (Identifier) the name of the DataSource to confirm exists
    // @param callback (Callback) Callback to fire when the source has been fetched.
    //                            Takes a single parameter <code>found</code> - true
    //                            if the Xml was found.
    // @visibility reifyOnSite
    //<
    
    hasDataSourceXml : function (dsName, callback) {
        if (this.dsDataSourceIsFileSource) {
            this.dsDataSource.hasFile({
                fileName: dsName,
                fileType: "ds",
                fileFormat: "xml"
            }, function (dsResponse, data, dsRequest) {
                // Return true if XML exists for dsName
                callback(data);
            }, {
                // DataSources are always shared across users
                operationId: "allOwners"
            });

        } else {
            var _this = this;
            this.dsDataSource.fetchData(
                {[this.dsDataSourceIDField]:dsName},
                function (dsResponse, data, dsRequest) {
                    var found = (dsResponse.totalRows > 0);
                    callback(found);
                },
                {startRow:0,endRow:1}
            );
        }
    },

    //> @method dataSourceEditorSession.getUniqueDataSourceName
    // Obtains a unique DataSource name in persistent storage.
    // By default, this performed against the +link{dsDataSource}.
    // Override this method to use a different persistent storage.
    //
    // @param baseDSName (Identifier) the base name of the new DataSource to find
    // @param callback (Callback) Callback to fire when the name has been determined.
    //                            Takes a single parameter <code>filename</code> - the
    //                            name of the new DataSource.
    // @visibility reifyOnSite
    //<
    getUniqueDataSourceName : function (baseDSName, callback) {
        // auto-generate an ID for new DataSource
        var fileSpec = {
            fileName: baseDSName,
            fileType: "ds",
            fileFormat: "xml",
            indexPrefix: "_"
        };
        this.dsDataSource.uniqueName(fileSpec, function (dsResponse, data, dsRequest) {
            if (data) {
                // Unique filename found
                callback(data.fileName);
                return;
            }
            // No unique filename found. This shouldn't happen but what is the recourse if it does?
            callback();
        }, {
            operationId: "uniqueNameNoAdd"
        });
    },

    loadDataSourceDefaults : function (dsName, callback) {
        this.fetchDataSourceJS(dsName, function (js) {
            var defaults = (js ? isc.DataSourceEditor.extractDSDefaultsFromJS(js) : null);
            callback(defaults);
        });
    },

    loadDataSourcesDefaults : function (dsNames, callback) {
        
        if (this.dsDataSourceIsFileSource) {
            const wasQueuing = isc.RPCManager.startQueue();
            
            const defaultsMap = {};    
            for (let i = 0; i < dsNames.length; i++) {
                const isLast = i == dsNames.length-1;
                const dsName = dsNames[i];
                this.fetchDataSourceJS(dsName, function (js) {

                    const defaults = (js ? isc.DataSourceEditor.extractDSDefaultsFromJS(js) : null);
                    defaultsMap[dsName] = defaults;

                    if (isLast) {
                        callback(defaultsMap);
                    }
                });
            }
            if (!wasQueuing) isc.RPCManager.sendQueue();

        
        } else {
            const wasQueuing = isc.RPCManager.startQueue(); // Queue for XML source
            //>DEBUG
            if (wasQueuing) {
                this.logWarn("DataSourceEditor.loadDataSourcesDefaults - sending existing RPCManager queue early to perform chained actions");
            }
            //<DEBUG
            const dsXMLMap = {};
            const _this = this;
            for (let i = 0; i < dsNames.length; i++) {
                const isLast = (i == dsNames.length-1);
                const dsName = dsNames[i];
                this.fetchDataSourceXml(dsName, 
                    function (xml) {
                        dsXMLMap[dsName] = xml;
                        if (isLast) {

                            const defaultsMap = {};
                            isc.RPCManager.startQueue();
                            let ii = 0;
                            for (let innerLoopDSName in dsXMLMap) {
                                const innerLoopIsLast = (ii == dsNames.length-1);
                                const innerLoopXML = dsXMLMap[innerLoopDSName];
                                ii++;
                                _this.dataSourceXMLToJS(
                                    innerLoopXML, 
                                    function (js) {
                                        const defaults = (js ? isc.DataSourceEditor.extractDSDefaultsFromJS(js) : null);
                                        defaultsMap[innerLoopDSName] = defaults;
                                        if (innerLoopIsLast) {
                                            // defaultsMap will be an object mapping dsNames to "defaults" blocks
                                            callback(defaultsMap);
                                        }
                                    }
                                );
                            }
                        }
                        isc.RPCManager.sendQueue();
                    }
                );
            }
            isc.RPCManager.sendQueue();
        }
    },

    convertDataSourcesDefaults : function (dsNodes, callback) {
        const wasQueuing = isc.RPCManager.startQueue();

        const defaultsMap = {};
        for (let i = 0; i < dsNodes.length; i++) {
            const isLast = (i === dsNodes.length-1);
            const dsNode = dsNodes[i];
            this.dataSourceXMLToJS(
                dsNode.xml,
                function (js) {
                    defaultsMap[dsNode.ID] = (js ? isc.DataSourceEditor.extractDSDefaultsFromJS(js) : null);
                    if (isLast) {
                        // defaultsMap will be an object mapping dsNames to "defaults" blocks
                        callback(defaultsMap);
                    }
                }
            );
        }
        if (!wasQueuing) isc.RPCManager.sendQueue();
    },

    createTempLiveDSInstance : function (defaults) {
        defaults = (defaults ? isc.clone(defaults) : {});
        // Configure DataSource to not register a global ID. Don't remove the ID
        // to assign an auto ID because the correct ID must be shown in the editor.
        defaults.addGlobalId = false;
        // Mark temp DataSource as if it was loaded from a sourceDataSource.
        // This satisfies the ID validator that this DS isn't a system DS.
        defaults.sourceDataSourceID = this.dsDataSource && this.dsDataSource.ID;
        defaults.fields = isc.clone(defaults.fields);

        // When creating a DataSource it will automatically be added to the DataSource
        // registrations so an isc.DS.get() will work but we don't want to overwrite
        // that either. Additionally, if the DS is found via isc.DS.get() during create
        // the previous instance will be destroyed.
        //
        // To avoid all of that, just stub out the getDataSource() and registerDataSource()
        // methods while creating the temp DataSource and put them back when done.
        var getDataSourceMethod = isc.DS.getDataSource,
            registerDataSourceMethod = isc.DS.registerDataSource
        ;
        isc.DS.addClassMethods({
            getDataSource : function (name, callback, context, schemaType) { },
            registerDataSource : function (dataSource) { }
        });

        // Create actual DS instance
        var ds = this.createLiveDSInstance(defaults);

        isc.DS.addClassMethods({
            getDataSource : getDataSourceMethod,
            registerDataSource : registerDataSourceMethod
        });

        return ds;
    },
    
    createLiveDSInstance : function (defaults) {
        var dsClass = this.dsClass || defaults._constructor || "DataSource",
            schema;
        if (isc.DS.isRegistered(dsClass)) {
            schema = isc.DS.get(dsClass);
        } else {
            schema = isc.DS.get("DataSource");
            defaults._constructor = dsClass;
        }
    
        // create a live instance
        var liveDS = isc.ClassFactory.getClass(dsClass).create(defaults);
        if (liveDS && this.dsDataSource) liveDS.sourceDataSourceID = this.dsDataSource.ID;

        return liveDS;
    },

    serializeDataSource : function (defaults, dsClass) {
        // handle custom subclasses of DataSource for which there is no schema defined by
        // serializing based on the DataSource schema but adding the _constructor property to
        // get the correct class.
        
        dsClass = dsClass || this.dsClass || defaults._constructor || "DataSource";
        var schema;
        if (isc.DS.isRegistered(dsClass)) {
            schema = isc.DS.get(dsClass);
        } else {
            schema = isc.DS.get("DataSource");
            defaults._constructor = dsClass;
        }
    
        // explicit class properties:
        // - in XML: "constructor" or xsi:type in instances, or "instanceConstructor" in schema
        // - for ClassFactory.newInstance(): _constructor
    
        // serialize to XML and save to server
        // - don't include any xsi:type attributes on field values.
        var xml = schema.xmlSerialize(defaults, { ignoreExplicitTypes: true });
        //this.logWarn("saving DS with XML: " + xml);
    
        return xml;
    },

    deleteRenamedDataSource : function (origID, origXml, deleteRenameCallback) {
        
        // Save original DS definition as a <dsID>_deleted record, marked as deleted.
        // How will DSEditor caller find out what the renamed file is called?
        var session = this;
        session.getUniqueDataSourceName(origID + isc._underscore + "deleted", function (filename) {
            // Save the original XML in the new "deleted" file
            session.saveDataSourceXml(filename, origXml, function (saveSuccess) {
                // Then remove the original DS file
                session.removeDataSourceXml(origID, function (removeSuccess) {
                    deleteRenameCallback(filename, saveSuccess && removeSuccess);
                });
            });
        });
    },
    
    fireDataSourceAdded : function (node) {
        this.fireCallback(this.dataSourceAdded, "name", [node.ID]);
    },
    dataSourceAdded : function (name) { },  // For observing

    fireDataSourceChanged : function (node) {
        this.fireCallback(this.dataSourceChanged, "name", [node.ID]);
    },
    dataSourceChanged : function (name) { },  // For observing

    fireDataSourceRenamed : function (node) {
        this.fireCallback(this.dataSourceRenamed, "fromName,toName", [node._renamedFrom,node.ID]);
    },
    dataSourceRenamed : function (name) { },  // For observing

    fireDataSourceDeleted : function (name) {
        this.fireCallback(this.dataSourceDeleted, "name", [name]);
    },
    dataSourceDeleted : function (name) { },  // For observing

    addChange : function (name, changeType, detail) {
        if (!this.useChangeTracking || this._revertingChange) return null;

        var changeTree = this._dsChanges,
            changeContext = this._dsChangeContext,
            parent = changeContext[changeContext.length-1] || changeTree.getRoot(),
            node = {
                dsName: name,
                changeType: changeType,
                description: this._getChangeDescription(name, changeType, detail),
                detail: detail
            }
        ;
        var newNode = changeTree.add(node, parent);
        
        return newNode;
    },

    startChangeContext : function (node) {
        if (!this.useChangeTracking || this._revertingChange) return;
        this._dsChangeContext.add(node);
    },

    endChangeContext : function (node) {
        if (!this.useChangeTracking || this._revertingChange) return;
        var changeContext = this._dsChangeContext;
        if (!node) {
            node = changeContext[changeContext.length-1]
        }
        var lastNode;
        do {
            lastNode = changeContext.pop();
        } while (lastNode && lastNode != node);
    },

    getChangeLog : function () {
        return this._dsChanges;
    },

    clearChangeLog : function () {
        this._dsChanges = isc.Tree.create();
        this._dsChangeContext = [];
    },

    revertChange : function (node, dsEditor, skipRefresh) {
        if (!this.useChangeTracking) return;
        var tree = this._dsChanges,
            children = (tree.getChildren(node) || []).duplicate(),
            allRevertedNodes = (!skipRefresh ? tree.getAllNodes(node) : null)
        ;
        if (!skipRefresh) {
            if (!allRevertedNodes) allRevertedNodes = [];
            allRevertedNodes.addAt(node, 0);
        }

        for (var i = 0; i < children.length; i++) {
            this.revertChange(children[i], dsEditor, true);
        }

        // Reverting typically uses the same methods as the original change. To prevent
        // the reverting change from being recorded as a new change, the _revertingChange
        // flag is set.
        this._revertingChange = true;

        var detail = node.detail;
        switch (node.changeType) {
            case "addField":
                this.removeField(node.dsName, detail.fieldName);
                break;
            case "changeFieldType":
                if (detail.oldType != null) {
                    this.changeFieldType(node.dsName, detail.fieldName, detail.oldType);
                }
                break;
            case "changeFieldProperty":
                if (detail.oldType != null) {
                    this.changeFieldProperty(node.dsName, detail.fieldName, detail.property, detail.originalValue);
                }
                break;
            default:
                this.logWarn("Cannot revert change: " + this.echoFull(tree.getCleanNodeData(node)));
                break;
        }
        tree.remove(node);

        delete this._revertingChange;

        if (dsEditor && !skipRefresh) {
            var updatedDSNames = allRevertedNodes.getProperty("dsName").getUniqueItems();
            for (var i = 0; i < updatedDSNames.length; i++) {
                var dsName = updatedDSNames[i];

                // The fields grid needs to be rebound
                var defaults = this.getDataSourceDefaults(dsName),
                    fields = defaults.fields
                ;

                // Re-bind defaults to FieldEditor
                dsEditor.bindFields(defaults);

                // Re-create testDS with updated fields/data
                if (dsEditor.editSampleData) {
                    dsEditor._rebindSampleDataGrid(fields, defaults.cacheData);
                }

                // Update selections and possibly value of titleField. Must come after sampleData updated
                dsEditor.updateTitleField();

                // Save editor changes (including the updated fields/sample data) to session
                // dsEditor.pushEditsToSession();
            }
        }
    },

    _changeDescriptionFormats: {
        "changeProperty": "${detail.newValue != null ? 'Change' : 'Remove'} ${dsName} property '${detail.property}' " +
            "${detail.newValue != null ? 'to ' + detail.newValue : ''}",
        "addField": "Add ${dsName} field '${detail.fieldName}' of type ${detail.field.type}${detail.field.length != null ? '(' + detail.field.length + ')' : ''}",
        "removeField": "Remove ${dsName} field '${detail.fieldName}'",
        "reorderField": "Moved ${dsName} field '${detail.fieldName}' from ${detail.fromPosition} '${detail.fromField}' to ${detail.toPosition} '${detail.toField}'",
        "changeFieldName": "Change ${dsName} field '${detail.fieldName}' to " + "'${detail.newName}'",
        "changeFieldType": "Change ${dsName} field '${detail.fieldName}' type from ${detail.oldType}${detail.oldLength != null ? '(' + detail.oldLength + ')' : ''} to ${detail.newType}${detail.newLength != null ? '(' + detail.newLength + ')' : ''}",
        "changeFieldProperty": "${detail.newValue != null ? 'Change' : 'Remove'} ${dsName} field '${detail.fieldName}' property '${detail.property}' " +
            "${detail.newValue != null ? 'to ' + detail.newValue : ''}",
        "addOperationBinding": "Add ${dsName} '${detail.operationType}${detail.operationId ? '/' + detail.operationId : ''}' operation binding",
        "removeOperationBinding": "Remove ${dsName} '${detail.operationType}${detail.operationId ? '/' + detail.operationId : ''}' operation binding",
        "changeOperationBindingProperty": "${detail.newValue != null ? 'Change' : 'Remove'} ${dsName} '${detail.operationType}${detail.operationId ? '/' + detail.operationId : ''}' operation binding property '${detail.property}' " +
            "${detail.newValue != null ? 'to ' + detail.newValue : ''}"
    },

    _getChangeDescription : function (dsName, changeType, detail) {
        var format = this._changeDescriptionFormats[changeType];
        return (format ? format.evalDynamicString(this, {
            dsName: dsName,
            detail: detail
        }) : null);
    },

    _changeTypes: {
        "changeProperty": { detailKeys: [ "property" ]},
        "changeFieldName": { detailKeys: [ "fieldName" ]},
        "changeFieldType": { detailKeys: [ "fieldName" ]},
        "changeFieldProperty": { detailKeys: [ "fieldName", "property" ]},
        "addField": { detailKeys: [ "fieldName" ]}
    },

    consolidateChanges : function () {
        if (!this.useChangeTracking) return [];
        // Simplify changes into a list removing intermediate changes
        var tree = this._dsChanges,
            allChanges = tree.getCleanNodeData(tree.getAllNodes(), false),
            consolidatedChanges = []
        ;

        // changeProperty
        //      dsName
        //      detail.property
        //      detail.originalValue
        //      detail.newValue

        // changeFieldName
        //      dsName
        //      detail.fieldName
        //      detail.newName

        // changeFieldType
        //      dsName
        //      detail.fieldName
        //      detail.oldType
        //      detail.newType
        //      detail.oldLength
        //      detail.newLength

        // changeFieldProperty
        //      dsName
        //      detail.fieldName
        //      detail.property
        //      detail.originalValue
        //      detail.newValue

        // addField
        //      dsName
        //      detail.fieldName
        //      detail.field

        // removeField
        //      dsName
        //      detail.fieldName
        //      detail.field

        // reorderField
        //      dsName
        //      detail.fieldName
        //      detail.fromPosition
        //      detail.fromField
        //      detail.toPosition
        //      detail.toField

        // addOperationBinding
        //      dsName
        //      detail.binding
    
        // removeOperationBinding
        //      dsName
        //      detail.binding

        // changeOperationBindingProperty
        //      dsName
        //      detail.operationType
        //      detail.operationId  (optional)
        //      detail.property
        //      detail.originalValue
        //      detail.newValue

        // ds.title

        // CFN: ds.title -> ds.title2
        //    { changeFieldName_ds_title2: { fieldName: <fn>, newName: <fn>} }
        // CFN: ds.title2 -> ds.title3
        //    

        for (var i = 0; i < allChanges.length; i++) {
            var change = allChanges[i],
                detail = change.detail
            ;

            switch (change.changeType) {
                case "changeProperty":
                    // Check for a previous change
                    var key = change.changeType + "_" + change.dsName + "_" + detail.property
                        oldChange = consolidatedChanges.find("key", key)
                    ;
                    if (!oldChange) {
                        consolidatedChanges.add({
                            key: key,
                            change: isc.clone(change)
                        });
                    } else if (oldChange.change.detail.originalValue == detail.newValue) {
                        consolidatedChanges.remove(oldChange);
                    } else {
                        var oldDetail = oldChange.change.detail;
                        oldDetail.newValue = detail.newValue;
                        oldChange.change.description = this._getChangeDescription(change.dsName, change.changeType, oldDetail);
                    }
                    break;
                case "addField":
                    var key = change.changeType + "_" + change.dsName + "_" + detail.fieldName
                        oldChange = consolidatedChanges.find("key", key)
                    ;
                    if (!oldChange) {
                        consolidatedChanges.add({
                            key: key,
                            change: isc.clone(change)
                        });
                    }
                    break;
                case "removeField":
                    var key = "addField_" + change.dsName + "_" + detail.fieldName
                        oldChange = consolidatedChanges.find("key", key)
                    ;
                    if (oldChange) {
                        consolidatedChanges.remove(oldChange);
                    } else {
                        var newKey = change.changeType + "_" + change.dsName + "_" + detail.fieldName;
                        consolidatedChanges.add({
                            key: newKey,
                            change: isc.clone(change)
                        });
                    }
                    break;
                case "reorderField":
                    var key = change.changeType + "_" + change.dsName + "_" + detail.fieldName
                        oldChange = consolidatedChanges.find("key", key)
                    ;
                    if (oldChange) {
                        var oldDetail = oldChange.change.detail;
                        oldDetail.position = detail.position;
                        oldDetail.targetField = detail.targetField;
                        oldChange.change.description = this._getChangeDescription(change.dsName, change.changeType, oldDetail);

                        // consolidatedChanges.remove(oldChange);
                    } else {
                        consolidatedChanges.add({
                            key: key,
                            change: isc.clone(change)
                        });
                    }
                    break;
                case "changeFieldName":
                    // Check for this change being made to a new field
                    var oldKey = "addField_" + change.dsName + "_" + detail.fieldName,
                        oldChange = consolidatedChanges.find("key", oldKey)
                    ;
                    if (oldChange) {
                        var newKey = "addField_" + change.dsName + "_" + detail.newName,
                            oldDetail = oldChange.change.detail,
                            field = isc.clone(oldDetail.field)
                        ;
                        field.name = detail.newName;
                        oldDetail.fieldName = field.name;
                        oldDetail.field = field;
                        oldChange.change.description = this._getChangeDescription(change.dsName, "addField", oldDetail);
                        oldChange.change.fieldName = detail.newName
                        oldChange.key = newKey;
                    } else {
                        // Now check for a previous change
                        var oldKey = change.changeType + "_" + change.dsName + "_" + detail.fieldName,
                            newKey = change.changeType + "_" + change.dsName + "_" + detail.newName,
                            oldChange = consolidatedChanges.find("key", oldKey)
                        ;
                        if (!oldChange) {
                            consolidatedChanges.add({
                                key: newKey,
                                change: isc.clone(change)
                            });
                        } else if (oldChange.change.detail.fieldName == detail.newName) {
                            consolidatedChanges.remove(oldChange);
                        } else {
                            var oldDetail = oldChange.change.detail;
                            oldDetail.newName = detail.newName;
                            oldChange.change.description = this._getChangeDescription(change.dsName, change.changeType, oldDetail);
                            oldChange.key = newKey;
                        }
                    }
                    break;
                case "changeFieldType":
                    // Check for this change being made to a new field
                    var key = "addField_" + change.dsName + "_" + detail.fieldName
                        oldChange = consolidatedChanges.find("key", key)
                    ;
                    if (oldChange) {
                        var oldDetail = oldChange.change.detail,
                            field = isc.clone(oldDetail.field)
                        ;
                        field.type = detail.newType;
                        field.length = detail.newLength;
                        oldDetail.field = field;
                    } else {
                        // Now check for a previous change
                        var key = change.changeType + "_" + change.dsName + "_" + detail.fieldName
                            oldChange = consolidatedChanges.find("key", key)
                        ;
                        if (!oldChange) {
                            consolidatedChanges.add({
                                key: key,
                                change: isc.clone(change)
                            });
                        } else if (oldChange.change.detail.oldType == detail.newType) {
                            consolidatedChanges.remove(oldChange);
                        } else {
                            var oldDetail = oldChange.change.detail;
                            oldDetail.newType = detail.newType;
                            oldDetail.newLength = detail.newLength;
                            oldChange.change.description = this._getChangeDescription(change.dsName, change.changeType, oldDetail);
                        }
                    }
                    break;
                case "changeFieldProperty":
                    // Check for this change being made to a new field
                    var key = "addField_" + change.dsName + "_" + detail.fieldName
                        oldChange = consolidatedChanges.find("key", key)
                    ;
                    if (oldChange) {
                        var oldDetail = oldChange.change.detail;
                        if (detail.newValue == null) {
                            delete oldDetail[detail.property];
                        } else {
                            oldDetail[detail.property] = detail.newValue;
                        }
                    } else {
                        // Now check for a previous change
                        var key = change.changeType + "_" + change.dsName + "_" + detail.fieldName + "_" + detail.property
                            oldChange = consolidatedChanges.find("key", key)
                        ;
                        if (!oldChange) {
                            consolidatedChanges.add({
                                key: key,
                                change: isc.clone(change)
                            });
                        } else if (oldChange.change.detail.oldType == detail.newType) {
                            consolidatedChanges.remove(oldChange);
                        } else {
                            var oldDetail = oldChange.change.detail;
                            oldDetail[detail.property] = detail.newValue;
                            oldChange.change.description = this._getChangeDescription(change.dsName, change.changeType, oldDetail);
                        }
                    }
                    break;
                case "addOperationBinding":
                    var key = change.changeType + "_" + change.dsName + "_" + detail.operationType + (detail.operationId ? "_" + detail.operationId : "")
                        oldChange = consolidatedChanges.find("key", key)
                    ;
                    if (!oldChange) {
                        consolidatedChanges.add({
                            key: key,
                            change: isc.clone(change)
                        });
                    }
                    break;
                case "removeOperationBinding":
                    var key = "addOperationBinding_" + change.dsName + "_" + detail.operationType + (detail.operationId ? "_" + detail.operationId : "")
                        oldChange = consolidatedChanges.find("key", key)
                    ;
                    if (oldChange) {
                        consolidatedChanges.remove(oldChange);
                    } else {
                        var newKey = change.changeType + "_" + change.dsName + "_" + detail.operationType + (detail.operationId ? "_" + detail.operationId : "");
                        consolidatedChanges.add({
                            key: newKey,
                            change: isc.clone(change)
                        });
                    }
                    break;
                case "changeOperationBindingProperty":
                    // Check for this change being made to a new binding
                    var key = "addOperationBinding_" + change.dsName + "_" + detail.operationType + (detail.operationId ? "_" + detail.operationId : "")
                        oldChange = consolidatedChanges.find("key", key)
                    ;
                    if (oldChange) {
                        var oldDetail = oldChange.change.detail;
                        if (detail.newValue == null) {
                            delete oldDetail[detail.property];
                        } else {
                            oldDetail[detail.property] = detail.newValue;
                        }
                    } else {
                        // Now check for a previous change
                        var key = change.changeType + "_" + change.dsName + "_" + detail.operationType + (detail.operationId ? "_" + detail.operationId : "") + "_" + detail.property
                            oldChange = consolidatedChanges.find("key", key)
                        ;
                        if (!oldChange) {
                            consolidatedChanges.add({
                                key: key,
                                change: isc.clone(change)
                            });
                        } else if (oldChange.change.detail.originalValue == detail.newValue) {
                            // Reverting a previously changed value: remove the earlier change
                            consolidatedChanges.remove(oldChange);
                        } else {
                            // Updating the same property as a previous change: update it in place
                            var oldDetail = oldChange.change.detail;
                            oldDetail.newValue = detail.newValue;
                            oldChange.change.description = this._getChangeDescription(change.dsName, change.changeType, oldDetail);
                        }
                    }
                    break;
                default:
                    break;
            }
        }
        return consolidatedChanges.getProperty("change");
    }
    
});

}