// ---- helpers ----
function base64url(input) {
  const utf8 = unescape(encodeURIComponent(input));
  const b64 = btoa(utf8);
  return b64.replace(/\+/g, "-").replace(/\//g, "_").replace(/=+$/g, "");
}

// Append ".parquet" if the URL path (before query) doesn't already end with it
function ensureParquetExtension(u) {
  if (!u) return u;
  const q = u.indexOf("?");
  const path = q >= 0 ? u.substring(0, q) : u;
  if (!/\.parquet$/i.test(path)) return u + ".parquet";
  return u;
}

// a ListGrid to display results
isc.ListGrid.create({
  ID: "listGridContent",
  width: "100%",
  height: 400,
  autoFetchData: true,
  showFilterEditor: true
});


isc.DynamicForm.create({
  ID: "parquetForm",
  width: "100%",
  numCols: 2,
  titleWidth: 200,
  fields: [
    {
      name: "datasets", title: "Parquet Datasets", width: "100%",
      editorType: "ComboBoxItem", defaultValue: "parquet-titanic",
      valueMap: {
        "parquet-titanic": "Titanic Passengers",
        "parquet-weather": "Weather Data",
        "parquet-mtcars": "Motor Trend Car Road Tests",
      }
    },
    {
      name: "custom_url", title: "Custom Dataset", type: "checkbox", value: false,
      changed: function (form, _, value) {
        // Toggle required & redraw to apply showIf/required
        const urlItem = form.getItem("parquet_custom_url");
        urlItem.setRequired(value);
        form.markForRedraw();
      }
    },
    {
      name: "parquet_custom_url", title: "URL parquet dataset", type: "text",
      showIf: "return form.getValue('custom_url') === true;",
      defaultValue: "https://www.tablab.app/sample-datasets/titanic.parquet",
      required: false,
      validators: [
        { type: "requiredIf", expression: "item.form.getValue('custom_url')===true" },
        { type: "regexp", expression: /^(https?:\/\/).+/i, errorMessage: "URL must start with http(s)://" }
      ],
      width: "100%"
    },
    {
      type: "button", title: "Load Data",
      click: function () {
        const useCustom = parquetForm.getValue("custom_url") === true;
        let dsID = null;

        if (useCustom) {
          if (!parquetForm.validate()) {
            isc.warn("Please provide a valid Parquet URL.");
            return;
          }
          let url = parquetForm.getValue("parquet_custom_url");
          if (!url || !url.trim()) {
            isc.warn("Custom URL is required when 'Custom Dataset' is checked.");
            return;
          }

          url = ensureParquetExtension(url.trim());
          dsID = "parquet-url-" + base64url(url);
        } else {
          const preset = parquetForm.getValue("datasets");
          if (!preset) {
            isc.warn("Please select a dataset.");
            return;
          }
          dsID = preset;
        }

        isc.DataSource.get(dsID, function (ds) {
          listGridContent.setDataSource(ds);
          listGridContent.fetchData();
        });
      }
    }
  ]
})

isc.Label.create({
  ID: "parquetInfo",
  contents: "You can load data from public parquet datasets files such as titanic Passengers, Motor Trend Car Road Tests, and Weather Data; you can also load your customr parquet urls",
  width: "100%",
  height: 12,
  autoDraw: true
});


// Now lay them out vertically
isc.VLayout.create({
  width: "100%", height: "100%",
  membersMargin: 10,    // space between form and grid
  members: [
    parquetForm,
    parquetInfo,
    listGridContent
  ]
});
